package ru.yandex.direct.core.entity.mobilecontent.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.repository.BsResyncQueueRepository;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.model.StatusIconModerate;
import ru.yandex.direct.core.entity.mobilecontent.model.StoreCountry;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.MobileContentMonitoringNotification;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static com.google.common.collect.ImmutableSet.toImmutableSet;
import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.StringConstraints.onlyUtf8Mb3Symbols;

/**
 * Сервис для обновления данных по мобильному контенту в таблице ppc.mobile_content.
 */
@Service
@ParametersAreNonnullByDefault
public class UpdateMobileContentService {
    private static final Logger logger = LoggerFactory.getLogger(UpdateMobileContentService.class);
    private static final String DEFAULT_COUNTRY_CODE = StoreCountry.RU.name().toLowerCase();
    static final String UNKNOWN_APP_NAME = "Unknown app name";
    /**
     * Размер чанков для вставки в очередь на отправку в БК изменённых баннеров.
     */
    static final int RESYNC_TO_BS_CHUNK_SIZE = 5000;

    public static final Set<ModelProperty> PROPERTIES_TO_UPDATE = ImmutableSet.of(
            MobileContent.AGE_LABEL,
            MobileContent.APP_SIZE,
            MobileContent.AVAILABLE_ACTIONS,
            MobileContent.BUNDLE_ID,
            MobileContent.DOWNLOADS,
            MobileContent.GENRE,
            MobileContent.ICON_HASH,
            MobileContent.MIN_OS_VERSION,
            MobileContent.NAME,
            MobileContent.PRICES,
            MobileContent.PUBLISHER_DOMAIN_ID,
            MobileContent.RATING,
            MobileContent.RATING_VOTES,
            MobileContent.SCREENS
    );
    // Поля, изменения которых приводит к сбросу statusBsSynced
    public static final Set<ModelProperty> BS_FIELDS = Set.of(
            MobileContent.AGE_LABEL,
            MobileContent.AVAILABLE_ACTIONS,
            MobileContent.BUNDLE_ID,
            MobileContent.DOWNLOADS,
            MobileContent.IS_AVAILABLE,
            MobileContent.MIN_OS_VERSION,
            MobileContent.NAME,
            MobileContent.PRICES,
            MobileContent.PUBLISHER_DOMAIN_ID,
            MobileContent.RATING,
            MobileContent.RATING_VOTES,
            MobileContent.SCREENS
    );
    private static final Set<String> LANGUAGES = EnumSet.allOf(StoreCountry.class)
            .stream()
            .map(Enum::toString)
            .map(String::toLowerCase)
            .collect(toImmutableSet());

    private final MobileContentRepository mobileContentRepository;
    private final MobileContentYtHelper mobileContentYtHelper;
    private final MobileContentService mobileContentService;
    private final BsResyncQueueRepository bsResyncQueueRepository;
    private final NotificationService notificationService;
    private final ShardHelper shardHelper;
    private final BannerModerationRepository bannerModerationRepository;

    @Autowired
    public UpdateMobileContentService(
            MobileContentRepository mobileContentRepository,
            MobileContentYtHelper mobileContentYtHelper,
            MobileContentService mobileContentService,
            BsResyncQueueRepository bsResyncQueueRepository,
            NotificationService notificationService,
            ShardHelper shardHelper,
            BannerModerationRepository bannerModerationRepository) {
        this.mobileContentRepository = mobileContentRepository;
        this.mobileContentYtHelper = mobileContentYtHelper;
        this.mobileContentService = mobileContentService;
        this.bsResyncQueueRepository = bsResyncQueueRepository;
        this.notificationService = notificationService;
        this.shardHelper = shardHelper;
        this.bannerModerationRepository = bannerModerationRepository;
    }

    /**
     * Обновляет записи заданных мобильных контентов для заданных клиентов в таблице ppc.mobile_content в соответствии
     * с данными из store.
     *
     * @param logins           логины клиентов, чьи мобильные контенты нужно обновить
     * @param mobileContentIds id мобильных контентов, записи которых нужно обновить
     * @param bannerIds        id баннеров, для которых нужно обновить мобильные контенты
     * @return мап шард -> список изменений в таблице mobile_content для данного шарда
     */
    public Map<Integer, List<AppliedChanges<MobileContent>>> process(List<String> logins, List<Long> mobileContentIds,
                                                                     List<Long> bannerIds) {
        List<Long> clientIds = StreamEx.of(shardHelper.getClientIdsByLogins(logins).values())
                .distinct()
                .toList();
        Map<Long, Integer> shardByClientId = shardHelper.getShardsByClientIds(clientIds);
        Map<Integer, Collection<Long>> clientIdsByShard = Multimaps.index(clientIds, shardByClientId::get).asMap();

        Map<Integer, List<AppliedChanges<MobileContent>>> appliedChangesByShard = new HashMap<>();

        clientIdsByShard.forEach((shard, clientIdsChunk) -> {
            Collection<Long> mobileContentIdsFromBanners =
                    mobileContentRepository.getMobileContentIdsByBannerIds(shard, bannerIds).values();

            List<AppliedChanges<MobileContent>> appliedChanges = new ArrayList<>();
            Map<OsType, List<MobileContent>> mobileContentsByOsType =
                    StreamEx.of(mobileContentRepository.getMobileContent(shard, clientIdsChunk,
                            StreamEx.of(mobileContentIds)
                                    .append(mobileContentIdsFromBanners)
                                    .distinct()
                                    .toList()))
                            .mapToEntry(MobileContent::getOsType, mc -> mc)
                            .collapseKeys()
                            .toMap();
            mobileContentsByOsType.forEach((osType, mobileContents) ->
                    appliedChanges.addAll(processChunk(shard, osType, mobileContents)));
            appliedChangesByShard.put(shard, appliedChanges);
        });
        return appliedChangesByShard;
    }

    /**
     * Обновляет записи заданных мобильных контентов в таблице ppc.mobile_content в соответствии с данными из store.
     *
     * @param shard  шард с таблицей, которую нужно обновить
     * @param osType тип ОС всех заданных мобильных контентов
     * @param chunk  объекты мобильных контентов, записи которых нужно обновить
     * @return список изменений мобильных контентов после обновления
     */
    public List<AppliedChanges<MobileContent>> processChunk(int shard, OsType osType, Collection<MobileContent> chunk) {
        // Технически одно приложение может быть привязано к разным клиентам, потому Multimap
        Multimap<YTreeMapNode, MobileContent> orig = ArrayListMultimap.create(chunk.size(), chunk.size());
        for (MobileContent mc : chunk) {
            String country = mc.getStoreCountry().toLowerCase();
            YTreeMapNode lookupKey = MobileContentYtHelper.createLookupKey(mc.getStoreContentId(),
                    LANGUAGES.contains(country) ? country : DEFAULT_COUNTRY_CODE);
            orig.put(lookupKey, mc);
        }
        LocalDateTime ytStoreQueryTime = LocalDateTime.now();
        Map<YTreeMapNode, MobileContent> fromYt = listToMap(
                mobileContentYtHelper.getMobileContentFromYt(shard, osType, orig.keySet()),
                mc -> MobileContentYtHelper.createLookupKey(mc.getStoreContentId(), mc.getStoreCountry().toLowerCase()),
                identity());

        List<AppliedChanges<MobileContent>> changes = collectChanges(orig, fromYt, ytStoreQueryTime);
        ignoreUnsupportedNames(changes);
        mobileContentService.updateMobileContent(shard, changes);
        resyncChangesToBs(shard, changes);
        sendAvailabilityNotifications(shard, changes);
        return changes;
    }

    /**
     * DIRECT-83863: Игнорирует изменение имен, которые не подпадают под utf8mb3
     */
    void ignoreUnsupportedNames(Collection<AppliedChanges<MobileContent>> changes) {
        for (AppliedChanges<MobileContent> change : changes) {
            String newName = change.getNewValue(MobileContent.NAME);
            String oldName = change.getOldValue(MobileContent.NAME);
            if (newName != null && onlyUtf8Mb3Symbols().apply(newName) != null) {
                change.modify(MobileContent.NAME, oldName != null ? oldName : UNKNOWN_APP_NAME);
            }
        }
    }

    /**
     * Отправляет набор уведомлений об изменении доступности контента в магазине.
     *
     * @param shard   шард
     * @param changes изменения мобильного контента
     */
    void sendAvailabilityNotifications(int shard, Collection<AppliedChanges<MobileContent>> changes) {
        Map<Long, Boolean> notifyChanged = StreamEx.of(changes)
                .filter(ac -> ac.changed(MobileContent.IS_AVAILABLE))
                .toMap(ac -> ac.getModel().getId(), ac -> ac.getNewValue(MobileContent.IS_AVAILABLE));
        Collection<MobileContentMonitoringNotification> notifications = mobileContentRepository
                .getContentAvailabilityNotifications(shard, notifyChanged);
        notifications.forEach(notificationService::addNotification);
    }

    /**
     * Добавляет в очередь на отправку в БК измененные объекты мобильного контента.
     *
     * @param shard   шард
     * @param changes изменения мобильного контента
     */
    void resyncChangesToBs(int shard, Collection<AppliedChanges<MobileContent>> changes) {
        List<Long> resyncIds = StreamEx.of(changes)
                .filter(ac -> ac.changed(MobileContent.MIN_OS_VERSION)
                        || ac.changed(MobileContent.PUBLISHER_DOMAIN_ID))
                .map(ac -> ac.getModel().getId())
                .toList();
        List<BsResyncItem> toResync = mobileContentRepository.getMobileContentForResync(shard, resyncIds);
        // Отправляем чанками, т.к. привяазанных баннеров может быть значительно больше, чем приложений.
        Iterables.partition(toResync, RESYNC_TO_BS_CHUNK_SIZE)
                .forEach(chunk -> bsResyncQueueRepository.addToResync(shard, chunk));
    }

    /**
     * Создает список {@link AppliedChanges} на основании разницы данных из базы и данных из Yt.
     *
     * @param fromDb           Данные из базы, априори считаются устаревшими
     * @param fromYt           Данные из Yt, априори считаются более новыми
     * @param ytStoreQueryTime Время получения данных из Yt
     */
    List<AppliedChanges<MobileContent>> collectChanges(Multimap<YTreeMapNode, MobileContent> fromDb,
                                                       Map<YTreeMapNode, MobileContent> fromYt,
                                                       LocalDateTime ytStoreQueryTime) {
        List<AppliedChanges<MobileContent>> changes = new ArrayList<>();
        // Обновление данных
        for (YTreeMapNode key : fromDb.keySet()) {
            for (MobileContent mcFromDb : fromDb.get(key)) {
                ModelChanges<MobileContent> mc = new ModelChanges<>(mcFromDb.getId(), MobileContent.class);
                MobileContent mcFromYt = fromYt.get(key);
                if (mcFromYt != null) {
                    PROPERTIES_TO_UPDATE.forEach(prop -> mc.process(prop.get(mcFromYt), prop));
                    mc.process(true, MobileContent.IS_AVAILABLE);
                    mc.process(0, MobileContent.TRIES_COUNT);
                } else {
                    // Не помечаем приложение как недоступное до трех дней от create_time
                    if (mcFromDb.getCreateTime().isBefore(LocalDateTime.now().minusDays(3))) {
                        mc.process(false, MobileContent.IS_AVAILABLE);
                        mc.process(mcFromDb.getTriesCount() + 1, MobileContent.TRIES_COUNT);
                        logger.warn("YT key {} not found", key);
                    }
                }
                changes.add(mc.applyTo(mcFromDb));
            }
        }

        // Обновление timestamp'ов
        changes.forEach(ac -> {
            if (ac.hasActuallyChangedProps()) {
                ac.modify(MobileContent.MODIFY_TIME, ytStoreQueryTime);

                // если данные обновились, то сбрасываем statusBsSynced
                if (BS_FIELDS.stream().anyMatch(ac::changed)) {
                    ac.modify(MobileContent.STATUS_BS_SYNCED, StatusBsSynced.NO);
                }

                // если обновилась иконка нужно сбросить статус модерации
                if (ac.changed(MobileContent.ICON_HASH)) {
                    ac.modify(MobileContent.STATUS_ICON_MODERATE, StatusIconModerate.READY);
                }
            }
            ac.modify(MobileContent.STORE_REFRESH_TIME, ytStoreQueryTime);
        });
        return changes;
    }
}
