package ru.yandex.direct.core.entity.mobilecontent.util;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.eclipse.jetty.util.StringUtil;

import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.model.ContentType;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.uac.model.Store;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.uac.validation.ParseAppInfoUrlDefectsKt.invalidAppId;
import static ru.yandex.direct.core.entity.uac.validation.ParseAppInfoUrlDefectsKt.invalidAppUrl;
import static ru.yandex.direct.core.entity.uac.validation.ParseAppInfoUrlDefectsKt.invalidGooglePlayUrl;
import static ru.yandex.direct.core.entity.uac.validation.ParseAppInfoUrlDefectsKt.invalidItunesUrl;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;


/**
 * Парсит URL cтора (Google Play или Apple App Store), извлекая из него информацию про тип стора, приложение,
 * страну, язык и категорию. Если удалось распарсить url, то возвращает {@code Optional} с {@code MobileAppStoreUrl},
 * иначе пустой {@code Optional}.
 * <p>
 * yandex-lib/mobile_redirect/lib/Yandex/CheckMobileRedirect.pm:373
 */
public class MobileAppStoreUrlParser {
    public static final String GOOGLE_PLAY_DOMAIN = "play.google.com";
    public static final String ITUNES_APP_STORE_OLD_DOMAIN = "itunes.apple.com";
    public static final String ITUNES_APP_STORE_DOMAIN = "apps.apple.com";
    private static final Map<String, ContentType> GOOGLE_PLAY_CONTENT_TYPES = ImmutableMap.of("apps", ContentType.APP);
    private static final Pattern GOOGLE_PLAY_CONTENT_ID = Pattern.compile(
            "^([a-z][a-z_0-9]*\\.)*[a-z][a-z_0-9]*$", Pattern.CASE_INSENSITIVE);
    private static final Pattern GOOGLE_PLAY_COUNTRY = Pattern.compile("^[a-z]{2}$", Pattern.CASE_INSENSITIVE);
    private static final Pattern GOOGLE_PLAY_LANGUAGE = Pattern.compile("^[a-z]{2}$", Pattern.CASE_INSENSITIVE);
    private static final Pattern APP_STORE_URL_PATTERN = Pattern.compile(
            "/+(?:([a-z]{2})/)?(app)/(?:.+/)?(id[0-9]+)/?", Pattern.CASE_INSENSITIVE);
    private static final Pattern GOOGLE_PLAY_URL_PATTERN = Pattern.compile(
            "/store/apps/details(/?|/[^/]+)$", Pattern.CASE_INSENSITIVE);

    private MobileAppStoreUrlParser() {
    }

    public static MobileAppStoreUrl parseStrictMemoized(@Nullable String url,
                                                        @Nonnull Map<String, MobileAppStoreUrl> resultMapping) {
        return resultMapping.computeIfAbsent(url, MobileAppStoreUrlParser::parseStrict);
    }

    public static MobileAppStoreUrl parseStrict(@Nullable String url) {
        Optional<MobileAppStoreUrl> optionalMobileAppStoreUrl = parse(url);
        checkState(optionalMobileAppStoreUrl.isPresent());
        return optionalMobileAppStoreUrl.get();
    }

    public static Optional<MobileAppStoreUrl> parseMemoized(@Nullable String url,
                                                            @Nonnull Map<String, MobileAppStoreUrl> resultMapping) {
        return Optional.ofNullable(resultMapping.computeIfAbsent(url, t -> parse(t).orElse(null)));
    }

    @Nonnull
    public static ValidationResult<MobileAppStoreUrl, Defect> parseWithDefect(@Nullable String url, Defect<Void> defaultDefect) {
        if (StringUtil.isBlank(url)) {
            return ValidationResult.failed(null, defaultDefect);
        }
        Optional<URI> maybeUri = sanitizeUrl(url);
        if (!maybeUri.isPresent()) {
            return ValidationResult.failed(null, defaultDefect);
        }
        URI uri = maybeUri.get();

        OsType osType = null;
        Store store = null;
        ContentType contentType = null;
        String storeContentId = null;
        String storeCountry = null;
        String storeLanguage = null;
        boolean isDefaultCountry = false;

        if (GOOGLE_PLAY_DOMAIN.equals(uri.getHost())) {
            osType = OsType.ANDROID;
            store = Store.GOOGLE_PLAY;
            var params = paramsOf(uri);
            Matcher matcher = GOOGLE_PLAY_URL_PATTERN.matcher(uri.getPath());
            if (!matcher.matches()  || uri.getRawQuery() == null) {
                return ValidationResult.failed(null, invalidGooglePlayUrl());
            }

            contentType = StreamEx.of(uri.getPath().split("/"))
                    .map(GOOGLE_PLAY_CONTENT_TYPES::get)
                    .nonNull()
                    .findFirst()
                    .orElse(null);

            storeContentId = Optional.ofNullable(params.get("id"))
                    .filter(v -> GOOGLE_PLAY_CONTENT_ID.matcher(v).matches())
                    .orElse(null);

            if (storeContentId == null) {
                return ValidationResult.failed(null, invalidAppId());
            }

            storeCountry = Optional.ofNullable(params.get("gl"))
                    .filter(v -> GOOGLE_PLAY_COUNTRY.matcher(v).matches())
                    .map(String::toUpperCase)
                    .orElse(null);

            storeLanguage = Optional.ofNullable(params.get("hl"))
                    .filter(v -> GOOGLE_PLAY_LANGUAGE.matcher(v).matches())
                    .map(String::toUpperCase)
                    .orElse(MobileAppStoreUrl.DEFAULT_LANGUAGE);

            if (storeCountry == null) {
                storeCountry = MobileAppStoreUrl.DEFAULT_COUNTRY;
                isDefaultCountry = true;
            }

        } else if (ITUNES_APP_STORE_DOMAIN.equals(uri.getHost()) || ITUNES_APP_STORE_OLD_DOMAIN.equals(uri.getHost())) {
            osType = OsType.IOS;
            store = Store.ITUNES;
            var params = paramsOf(uri);

            Matcher matcher = APP_STORE_URL_PATTERN.matcher(uri.getPath());
            if (matcher.matches()) {
                contentType = ContentType.APP;
                storeContentId = matcher.group(3);

                storeLanguage = Optional.ofNullable(params.get("l"))
                        .map(String::toUpperCase)
                        .orElse(MobileAppStoreUrl.DEFAULT_LANGUAGE);

                String country = ifNotNull(matcher.group(1), String::toUpperCase);
                if (country != null) {
                    if (country.equals("EN")) {
                        storeCountry = "US";
                    } else {
                        storeCountry = country.toUpperCase();
                    }
                } else {
                    storeCountry = MobileAppStoreUrl.DEFAULT_COUNTRY;
                    isDefaultCountry = true;
                }
            } else {
                return ValidationResult.failed(null, invalidItunesUrl());
            }
        }

        if (osType != null && contentType != null && storeContentId != null) {
            return ValidationResult.success(new MobileAppStoreUrl(
                    osType, store, contentType, storeCountry, storeLanguage, storeContentId, isDefaultCountry));
        } else {
            return ValidationResult.failed(null, defaultDefect);
        }
    }

    @Nonnull
    public static Optional<MobileAppStoreUrl> parse(@Nullable String url) {
        ValidationResult<MobileAppStoreUrl, Defect> parseResult = parseWithDefect(url, invalidAppUrl());
        if (parseResult.hasAnyErrors()) {
            return Optional.empty();
        }
        return Optional.of(parseResult.getValue());
    }

    /**
     * Выполняет предобработку и разбирает url
     */
    @Nonnull
    public static Optional<URI> sanitizeUrl(String url) {
        url = url.trim().replaceAll("^\"|\"$", "");

        URI uri;
        try {
            uri = new URI(url);
            if (uri.getScheme() == null) {
                uri = new URI("http://" + url);
            }
        } catch (URISyntaxException e) {
            return Optional.empty();
        }
        return Optional.of(uri);
    }

    private static Map<String, String> paramsOf(@Nonnull URI uri) {
        if (uri.getRawQuery() == null) {
            return Map.of();
        }

        Map<String, List<String>> params = StreamEx.of(uri.getRawQuery().split("&"))
                .remove(StringUtil::isBlank)
                .map(v -> v.split("=", 2))
                .mapToEntry(v -> v[0], v -> v.length < 2 ? "" : v[1])
                .mapKeys(MobileAppStoreUrlParser::decode)
                .mapValues(MobileAppStoreUrlParser::decode)
                .grouping();

        return EntryStream.of(params)
                .mapValues(l -> l.size() > 0 ? l.get(0) : null)
                .nonNullValues()
                .toMap();
    }

    private static String decode(@Nonnull String s) {
        try {
            return URLDecoder.decode(s, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new UnsupportedEncodingRuntimeException(
                    String.format("Can not decode from utf-8 string: '%s'", s), e);
        }
    }

    public static class UnsupportedEncodingRuntimeException extends UnsupportedOperationException {
        UnsupportedEncodingRuntimeException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}
