package ru.yandex.direct.core.entity.mobilegoals

import org.springframework.stereotype.Service

import ru.yandex.direct.core.entity.mobileapp.model.ExternalTrackerEventName
import ru.yandex.direct.core.entity.mobileapp.model.MobileExternalTrackerEvent;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppConversionStatisticRepository
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;

@Service
class MobileAppAutoGoalsService(
    private val mobileAppConversionStatisticRepository: MobileAppConversionStatisticRepository,
    private val mobileAppRepository: MobileAppRepository,
    private val mobileAppGoalsExternalTrackerRepository: MobileAppGoalsExternalTrackerRepository,
) {

    companion object {
        private val GOAL_ID_TO_EVENT_NAME = mapOf(
            38402972L to ExternalTrackerEventName.ACHIEVED_LEVEL,
            38403053L to ExternalTrackerEventName.ADDED_PAYMENT_INFO,
            38403071L to ExternalTrackerEventName.ADDED_TO_CART,
            38403080L to ExternalTrackerEventName.ADDED_TO_WISHLIST,
            38403008L to ExternalTrackerEventName.APP_LAUNCHED,
            38403095L to ExternalTrackerEventName.COMPLETED_REGISTRATION,
            38403104L to ExternalTrackerEventName.COMPLETED_TUTORIAL,
            38403530L to ExternalTrackerEventName.EVENT_1,
            38403545L to ExternalTrackerEventName.EVENT_2,
            38403581L to ExternalTrackerEventName.EVENT_3,
            38403131L to ExternalTrackerEventName.INITIATED_CHECKOUT,
            38403173L to ExternalTrackerEventName.PURCHASED,
            38403191L to ExternalTrackerEventName.RATED,
            38403197L to ExternalTrackerEventName.SEARCHED,
            38403494L to ExternalTrackerEventName.SHARED,
            38403206L to ExternalTrackerEventName.SPENT_CREDITS,
            38403338L to ExternalTrackerEventName.SPENT_TIME_IN_APP,
            38403215L to ExternalTrackerEventName.UNLOCKED_ACHIEVEMENT,
            38403230L to ExternalTrackerEventName.VIEWED_CONTENT,
        )
    }

    /**
     * Возвращает список целей для добавления сформированный на основе статистики конверсий DirectMobileAppStat
     *
     * @param shard шард
     * @param daysChecked количество дней для проверки в таблице DirectMobileAppStat
     */
    fun getAutoGoalsToAdd(
        shard: Int,
        daysChecked: Int,
    ) : List<MobileExternalTrackerEvent> {
        val conversionStatsByAppId = mobileAppConversionStatisticRepository.getConversionStatsByAppId(daysChecked)
        val mobileApps = mobileAppRepository.getMobileAppsWithoutRelations(shard, conversionStatsByAppId.keys)
        val mobileAppIds = mobileApps
            .map { it.getId() }
        val eventsByAppId = mobileAppGoalsExternalTrackerRepository.getEventsByAppIds(shard, mobileAppIds, false)
            .groupBy { it.getMobileAppId() }
        val clientIds = mobileApps
            .map { it.getClientId() }
            .toSet()
        val eventsToAdd = mutableListOf<MobileExternalTrackerEvent>()

        for (mobileApp in mobileApps) {
            val appStats = conversionStatsByAppId.getOrDefault(mobileApp.getId(), listOf())
            val appEventNames = eventsByAppId.getOrDefault(mobileApp.getId(), listOf())
                .map { it.getEventName() }
                .toSet()
            for (stats in appStats) {
                val goalId = stats.goalId
                if (!GOAL_ID_TO_EVENT_NAME.containsKey(goalId)) {
                    continue
                }
                val eventName = GOAL_ID_TO_EVENT_NAME.get(goalId)
                if (appEventNames.contains(eventName)) {
                    continue
                }
                eventsToAdd.add(MobileExternalTrackerEvent()
                    .withMobileAppId(mobileApp.getId())
                    .withEventName(eventName)
                    .withCustomName("")
                    .withIsDeleted(false)
                )
            }
        }
        return eventsToAdd
    }
}
