package ru.yandex.direct.core.entity.mobilegoals;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.mobileapp.model.AppmetrikaEventSubtype;
import ru.yandex.direct.core.entity.mobileapp.model.AppmetrikaEventType;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppMetrikaEvent;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppStoreType;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.dbschema.ppc.enums.MobileAppsStoreType;
import ru.yandex.direct.dbschema.ppcdict.enums.MobileAppGoalsAppmetrikaStoreType;
import ru.yandex.direct.dbschema.ppcdict.tables.records.MobileAppGoalsAppmetrikaRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppcdict.tables.MobileAppGoalsAppmetrika.MOBILE_APP_GOALS_APPMETRIKA;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@ParametersAreNonnullByDefault
@Repository
public class MobileAppGoalsAppmetrikaRepository {
    private static final JooqMapperWithSupplier<MobileAppMetrikaEvent> MAPPER = createMapper();

    private static JooqMapperWithSupplier<MobileAppMetrikaEvent> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(MobileAppMetrikaEvent::new)
                .map(property(MobileAppMetrikaEvent.ID, MOBILE_APP_GOALS_APPMETRIKA.GOAL_ID))
                .map(property(MobileAppMetrikaEvent.APP_METRIKA_APP_ID,
                        MOBILE_APP_GOALS_APPMETRIKA.APP_METRIKA_APPLICATION_ID))
                .map(property(MobileAppMetrikaEvent.BUNDLE_ID, MOBILE_APP_GOALS_APPMETRIKA.BUNDLE_ID))
                .map(convertibleProperty(MobileAppMetrikaEvent.STORE_TYPE,
                        MOBILE_APP_GOALS_APPMETRIKA.STORE_TYPE,
                        value -> MobileAppStoreType.fromSource(MobileAppsStoreType.valueOf(value.name())),
                        value -> MobileAppGoalsAppmetrikaStoreType.valueOf(MobileAppStoreType.toSource(value).name())))
                .map(convertibleProperty(MobileAppMetrikaEvent.EVENT_TYPE,
                        MOBILE_APP_GOALS_APPMETRIKA.EVENT_TYPE,
                        AppmetrikaEventType::fromSource,
                        AppmetrikaEventType::toSource))
                .map(convertibleProperty(MobileAppMetrikaEvent.EVENT_SUBTYPE,
                        MOBILE_APP_GOALS_APPMETRIKA.EVENT_SUBTYPE,
                        AppmetrikaEventSubtype::fromSource,
                        AppmetrikaEventSubtype::toSource))
                .map(property(MobileAppMetrikaEvent.EVENT_NAME, MOBILE_APP_GOALS_APPMETRIKA.EVENT_NAME))
                .map(property(MobileAppMetrikaEvent.CUSTOM_NAME, MOBILE_APP_GOALS_APPMETRIKA.CUSTOM_NAME))
                .map(booleanProperty(MobileAppMetrikaEvent.IS_DELETED, MOBILE_APP_GOALS_APPMETRIKA.IS_DELETED))
                .build();
    }

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public MobileAppGoalsAppmetrikaRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    public void addEvents(DSLContext dslContext, Collection<MobileAppMetrikaEvent> mobileMobileAppMetrikaEvents) {
        if (mobileMobileAppMetrikaEvents.isEmpty()) {
            return;
        }
        List<Long> ids = shardHelper.generateMobileAppGoalIds(mobileMobileAppMetrikaEvents.size());
        StreamEx.of(mobileMobileAppMetrikaEvents).zipWith(ids.stream())
                .forKeyValue(MobileAppMetrikaEvent::setId);

        new InsertHelper<>(dslContext, MOBILE_APP_GOALS_APPMETRIKA)
                .addAll(MAPPER, mobileMobileAppMetrikaEvents)
                .executeIfRecordsAdded();
    }

    public void addEvents(Collection<MobileAppMetrikaEvent> mobileMobileAppMetrikaEvents) {
        addEvents(dslContextProvider.ppcdict(), mobileMobileAppMetrikaEvents);
    }

    public void updateEvents(DSLContext dslContext, List<AppliedChanges<MobileAppMetrikaEvent>> changes) {
        JooqUpdateBuilder<MobileAppGoalsAppmetrikaRecord, MobileAppMetrikaEvent> ub =
                new JooqUpdateBuilder<>(MOBILE_APP_GOALS_APPMETRIKA.GOAL_ID, changes);
        ub.processProperty(MobileAppMetrikaEvent.CUSTOM_NAME, MOBILE_APP_GOALS_APPMETRIKA.CUSTOM_NAME);
        ub.processProperty(MobileAppMetrikaEvent.IS_DELETED, MOBILE_APP_GOALS_APPMETRIKA.IS_DELETED,
                RepositoryUtils::booleanToLong);
        dslContext
                .update(MOBILE_APP_GOALS_APPMETRIKA)
                .set(ub.getValues())
                .where(MOBILE_APP_GOALS_APPMETRIKA.GOAL_ID.in(ub.getChangedIds()))
                .execute();
    }

    public List<MobileAppMetrikaEvent> getEventsByMobileApps(Collection<MobileApp> mobileApps, boolean onlyActive) {
        var rows = mobileApps.stream()
                .filter(m -> m.getAppMetrikaApplicationId() != null)
                .map(mobileApp -> row(mobileApp.getAppMetrikaApplicationId(),
                        MobileContentService.getStoreAppIdFromMobileContent(mobileApp.getMobileContent()),
                        MobileAppGoalsAppmetrikaStoreType.valueOf(
                                MobileAppStoreType.toSource(mobileApp.getStoreType()).name())))
                .collect(toList());

        if (rows.isEmpty()) {
            return emptyList();
        }

        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(MOBILE_APP_GOALS_APPMETRIKA)
                .where(row(MOBILE_APP_GOALS_APPMETRIKA.APP_METRIKA_APPLICATION_ID,
                        MOBILE_APP_GOALS_APPMETRIKA.BUNDLE_ID,
                        MOBILE_APP_GOALS_APPMETRIKA.STORE_TYPE).in(rows))
                .and(onlyActive ? MOBILE_APP_GOALS_APPMETRIKA.IS_DELETED.eq(0L) : DSL.trueCondition())
                .orderBy(MOBILE_APP_GOALS_APPMETRIKA.GOAL_ID)
                .fetch(MAPPER::fromDb);
    }

    public List<MobileAppMetrikaEvent> getEventsByIds(Collection<Long> ids) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(MOBILE_APP_GOALS_APPMETRIKA)
                .where(MOBILE_APP_GOALS_APPMETRIKA.GOAL_ID.in(ids))
                .fetch(MAPPER::fromDb);
    }
}
