package ru.yandex.direct.core.entity.mobilegoals;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.mobileapp.model.ExternalTrackerEventName;
import ru.yandex.direct.core.entity.mobileapp.model.MobileExternalTrackerEvent;
import ru.yandex.direct.dbschema.ppc.tables.records.MobileAppGoalsExternalTrackerRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppc.tables.MobileAppGoalsExternalTracker.MOBILE_APP_GOALS_EXTERNAL_TRACKER;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@ParametersAreNonnullByDefault
@Repository
public class MobileAppGoalsExternalTrackerRepository {

    private static final JooqMapperWithSupplier<MobileExternalTrackerEvent> MAPPER = createMapper();

    private static JooqMapperWithSupplier<MobileExternalTrackerEvent> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(MobileExternalTrackerEvent::new)
                .map(property(MobileExternalTrackerEvent.ID, MOBILE_APP_GOALS_EXTERNAL_TRACKER.GOAL_ID))
                .map(property(MobileExternalTrackerEvent.MOBILE_APP_ID,
                        MOBILE_APP_GOALS_EXTERNAL_TRACKER.MOBILE_APP_ID))
                .map(convertibleProperty(MobileExternalTrackerEvent.EVENT_NAME,
                        MOBILE_APP_GOALS_EXTERNAL_TRACKER.EVENT_NAME,
                        ExternalTrackerEventName::valueOf,
                        ExternalTrackerEventName::toString))
                .map(property(MobileExternalTrackerEvent.CUSTOM_NAME, MOBILE_APP_GOALS_EXTERNAL_TRACKER.CUSTOM_NAME))
                .map(booleanProperty(MobileExternalTrackerEvent.IS_DELETED,
                        MOBILE_APP_GOALS_EXTERNAL_TRACKER.IS_DELETED))
                .build();
    }

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public MobileAppGoalsExternalTrackerRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    public void addEvents(DSLContext dslContext,
                          Collection<MobileExternalTrackerEvent> mobileMobileExternalTrackerEvents) {
        if (mobileMobileExternalTrackerEvents.isEmpty()) {
            return;
        }
        List<Long> ids = shardHelper.generateMobileAppGoalIds(mobileMobileExternalTrackerEvents.size());
        StreamEx.of(mobileMobileExternalTrackerEvents).zipWith(ids.stream())
                .forKeyValue(MobileExternalTrackerEvent::setId);

        new InsertHelper<>(dslContext, MOBILE_APP_GOALS_EXTERNAL_TRACKER)
                .addAll(MAPPER, mobileMobileExternalTrackerEvents)
                .executeIfRecordsAdded();
    }

    public void addEvents(int shard,
                          Collection<MobileExternalTrackerEvent> mobileMobileExternalTrackerEvents) {
        addEvents(dslContextProvider.ppc(shard), mobileMobileExternalTrackerEvents);
    }

    public List<MobileExternalTrackerEvent> getEventsByAppIds(DSLContext dslContext,
                                                              Collection<Long> mobileAppIds, boolean onlyActive) {
        return dslContext
                .select(MAPPER.getFieldsToRead())
                .from(MOBILE_APP_GOALS_EXTERNAL_TRACKER)
                .where(MOBILE_APP_GOALS_EXTERNAL_TRACKER.MOBILE_APP_ID.in(mobileAppIds)
                        .and(onlyActive ? MOBILE_APP_GOALS_EXTERNAL_TRACKER.IS_DELETED.eq(0L) : DSL.trueCondition()))
                .fetch(MAPPER::fromDb);
    }

    public List<MobileExternalTrackerEvent> getEventsByAppIds(int shard, Collection<Long> mobileAppIds,
                                                              boolean onlyActive) {
        return getEventsByAppIds(dslContextProvider.ppc(shard), mobileAppIds, onlyActive);
    }

    public List<MobileExternalTrackerEvent> getEventsByIds(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(MAPPER.getFieldsToRead())
                .from(MOBILE_APP_GOALS_EXTERNAL_TRACKER)
                .where(MOBILE_APP_GOALS_EXTERNAL_TRACKER.GOAL_ID.in(ids))
                .fetch(MAPPER::fromDb);
    }

    public void updateEvents(DSLContext dslContext, List<AppliedChanges<MobileExternalTrackerEvent>> changes) {
        if (changes.isEmpty()) {
            return;
        }
        JooqUpdateBuilder<MobileAppGoalsExternalTrackerRecord, MobileExternalTrackerEvent> ub =
                new JooqUpdateBuilder<>(MOBILE_APP_GOALS_EXTERNAL_TRACKER.GOAL_ID, changes);
        ub.processProperty(MobileExternalTrackerEvent.CUSTOM_NAME, MOBILE_APP_GOALS_EXTERNAL_TRACKER.CUSTOM_NAME);
        ub.processProperty(MobileExternalTrackerEvent.IS_DELETED, MOBILE_APP_GOALS_EXTERNAL_TRACKER.IS_DELETED,
                RepositoryUtils::booleanToLong);
        dslContext.update(MOBILE_APP_GOALS_EXTERNAL_TRACKER)
                .set(ub.getValues())
                .where(MOBILE_APP_GOALS_EXTERNAL_TRACKER.GOAL_ID.in(ub.getChangedIds()))
                .execute();
    }
}
