package ru.yandex.direct.core.entity.mobilegoals;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.mobileapp.model.AppmetrikaUniqueKey;
import ru.yandex.direct.core.entity.mobileapp.model.ExternalTrackerEventName;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppMetrikaEvent;
import ru.yandex.direct.core.entity.mobileapp.model.MobileExternalTrackerEvent;
import ru.yandex.direct.core.entity.mobileapp.util.MobileAppUtil;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.core.entity.mobilegoals.model.AppmetrikaInternalEvent;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static java.util.Collections.singleton;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.apache.commons.lang3.StringUtils.defaultIfBlank;
import static ru.yandex.direct.core.entity.mobileapp.model.AppmetrikaEventType.CLIENT;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class MobileAppGoalsService {
    private final MobileAppGoalsExternalTrackerRepository mobileAppGoalsExternalTrackerRepository;
    private final MobileAppGoalsAppmetrikaRepository mobileAppGoalsMetrikaRepository;
    private final DslContextProvider dslContextProvider;
    private final FeatureService featureService;
    private final ShardHelper shardHelper;

    public MobileAppGoalsService(MobileAppGoalsExternalTrackerRepository mobileAppGoalsExternalTrackerRepository,
                                 MobileAppGoalsAppmetrikaRepository mobileAppGoalsMetrikaRepository,
                                 DslContextProvider dslContextProvider, FeatureService featureService,
                                 ShardHelper shardHelper) {
        this.mobileAppGoalsExternalTrackerRepository = mobileAppGoalsExternalTrackerRepository;
        this.mobileAppGoalsMetrikaRepository = mobileAppGoalsMetrikaRepository;
        this.dslContextProvider = dslContextProvider;
        this.featureService = featureService;
        this.shardHelper = shardHelper;
    }

    /**
     * Добавляет цели по указанным событиям внешних трекеров
     * Если событие было, но его нет в указанных, то ему ставится признак is_deleted
     * При добавлении события, которое раньше было помечено is_deleted, используется эта запись, пометка снимается
     * Тесты сделаны на уровне контроллера - MobileAppControllerBaseTest
     */
    public void updateMobileAppGoalsForExternalTracker(DSLContext dslContext, ClientId clientId,
                                                       List<MobileApp> mobileApps) {
        if (!featureService.isEnabledForClientId(clientId, FeatureName.IN_APP_MOBILE_TARGETING)) {
            return;
        }

        if (featureService.isEnabledForClientId(clientId,
                FeatureName.IN_APP_MOBILE_TARGETING_CUSTOM_EVENTS_FOR_EXTERNAL_TRACKERS)) {
            List<MobileExternalTrackerEvent> eventsToAdd = new ArrayList<>();
            List<AppliedChanges<MobileExternalTrackerEvent>> changes = new ArrayList<>();
            for (MobileApp mobileApp : mobileApps) {
                if (mobileApp.getMobileExternalTrackerEvents() != null) {
                    Map<ExternalTrackerEventName, MobileExternalTrackerEvent> currentEventsByName
                            = listToMap(mobileAppGoalsExternalTrackerRepository.getEventsByAppIds(dslContext,
                            singleton(mobileApp.getId()), false), MobileExternalTrackerEvent::getEventName);

                    for (MobileExternalTrackerEvent event : mobileApp.getMobileExternalTrackerEvents()) {
                        MobileExternalTrackerEvent currentEvent = currentEventsByName.get(event.getEventName());
                        if (currentEvent == null) {
                            eventsToAdd.add(event.withMobileAppId(mobileApp.getId()));
                        } else {
                            changes.add(new ModelChanges<>(currentEvent.getId(), MobileExternalTrackerEvent.class)
                                    .process(event.getCustomName(), MobileExternalTrackerEvent.CUSTOM_NAME)
                                    .process(event.getIsDeleted(), MobileExternalTrackerEvent.IS_DELETED)
                                    .applyTo(currentEvent));

                            currentEventsByName.remove(event.getEventName());
                        }
                    }

                    for (MobileExternalTrackerEvent event : currentEventsByName.values()) {
                        changes.add(new ModelChanges<>(event.getId(), MobileExternalTrackerEvent.class)
                                .process(true, MobileExternalTrackerEvent.IS_DELETED)
                                .applyTo(event));
                    }
                }
            }

            mobileAppGoalsExternalTrackerRepository.addEvents(dslContext, eventsToAdd);
            mobileAppGoalsExternalTrackerRepository.updateEvents(dslContext, changes);
        } else {
            // Если фича IN_APP_MOBILE_TARGETING_CUSTOM_EVENTS_FOR_EXTERNAL_TRACKERS выключена, то создаются двадцать
            // целей по каждому значению из  ExternalTrackerEventName
            // todo удалить после готовности фронта в https://st.yandex-team.ru/DIRECT-143135 вместе с фичей
            // IN_APP_MOBILE_TARGETING_CUSTOM_EVENTS_FOR_EXTERNAL_TRACKERS

            List<Long> mapIds = mapList(mobileApps, MobileApp::getId);
            Set<Long> mapIdsWithGoals
                    = listToSet(mobileAppGoalsExternalTrackerRepository.getEventsByAppIds(dslContext, mapIds, false),
                    MobileExternalTrackerEvent::getMobileAppId);

            List<MobileExternalTrackerEvent> mobileGoals = StreamEx.of(mapIds)
                    .filter(id -> !mapIdsWithGoals.contains(id))
                    .cross(ExternalTrackerEventName.values())
                    .mapKeyValue((id, en) -> new MobileExternalTrackerEvent()
                            .withMobileAppId(id)
                            .withEventName(en)
                            .withCustomName("")
                            .withIsDeleted(false)
                    )
                    .collect(toList());
            mobileAppGoalsExternalTrackerRepository.addEvents(dslContext, mobileGoals);
        }
    }

    /**
     * Добавляет цели или ставит признак is_deleted по указанным событиям аппметрики.
     * При добавлении события, которое раньше было помечено is_deleted, используется эта запись, пометка снимается
     * Тесты сделаны на уровне контроллера - MobileAppControllerBaseTest
     */
    public void updateMobileAppGoalsForAppmetrika(ClientId clientId, List<MobileApp> mobileApps) {
        if (!featureService.isEnabledForClientId(clientId, FeatureName.IN_APP_MOBILE_TARGETING)) {
            return;
        }

        List<MobileAppMetrikaEvent> eventsToAdd = new ArrayList<>();
        List<AppliedChanges<MobileAppMetrikaEvent>> changes = new ArrayList<>();
        for (MobileApp mobileApp : mobileApps) {
            if (mobileApp.getMobileAppMetrikaEvents() != null) {
                String bundleId = MobileContentService.getStoreAppIdFromMobileContent(mobileApp.getMobileContent());
                Map<MobileAppMetrikaEvent, MobileAppMetrikaEvent> currentEventsByName
                        = listToMap(mobileAppGoalsMetrikaRepository.getEventsByMobileApps(singleton(mobileApp), false),
                        this::getKeyFieldsMobileAppMetrikaEvent
                );

                for (MobileAppMetrikaEvent event : mobileApp.getMobileAppMetrikaEvents()) {
                    event
                            .withBundleId(bundleId)
                            .withStoreType(mobileApp.getStoreType());

                    MobileAppMetrikaEvent key = getKeyFieldsMobileAppMetrikaEvent(event);
                    MobileAppMetrikaEvent currentEvent = currentEventsByName.get(key);
                    if (currentEvent == null) {
                        eventsToAdd.add(event);
                    } else {
                        changes.add(new ModelChanges<>(currentEvent.getId(), MobileAppMetrikaEvent.class)
                                .process(event.getCustomName(), MobileAppMetrikaEvent.CUSTOM_NAME)
                                .process(event.getIsDeleted(), MobileAppMetrikaEvent.IS_DELETED)
                                .applyTo(currentEvent));

                        currentEventsByName.remove(key);
                    }
                }

                for (MobileAppMetrikaEvent event : currentEventsByName.values()) {
                    changes.add(new ModelChanges<>(event.getId(), MobileAppMetrikaEvent.class)
                            .process(true, MobileAppMetrikaEvent.IS_DELETED)
                            .applyTo(event));
                }
            }
        }

        dslContextProvider.ppcdictTransaction(conf -> {
            DSLContext dslContext = conf.dsl();
            mobileAppGoalsMetrikaRepository.addEvents(dslContext, eventsToAdd);
            mobileAppGoalsMetrikaRepository.updateEvents(dslContext, changes);
        });
    }

    private MobileAppMetrikaEvent getKeyFieldsMobileAppMetrikaEvent(MobileAppMetrikaEvent e) {
        return new MobileAppMetrikaEvent()
                .withBundleId(e.getBundleId())
                .withStoreType(e.getStoreType())
                .withEventType(e.getEventType())
                .withEventSubtype(e.getEventSubtype())
                .withEventName(e.getEventName());
    }

    public List<Goal> getGoalsByApps(ClientId clientId, List<MobileApp> mobileApps) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        //внешние трекеры
        Map<Long, MobileApp> mobileAppMapsById = listToMap(mobileApps, MobileApp::getId);
        List<Goal> goals = mapList(mobileAppGoalsExternalTrackerRepository
                        .getEventsByAppIds(shard, mobileAppMapsById.keySet(), true),
                e -> externalTrackerEventToGoal(e, mobileAppMapsById));

        //аппметрика
        Map<AppmetrikaUniqueKey, MobileApp> mobileAppsByKey = mobileApps.stream()
                .filter(m -> m.getAppMetrikaApplicationId() != null)
                .collect(toMap(MobileAppUtil::getAppmetrikaUniqueKey, identity(),
                                (a, b) -> a.getId() < b.getId() ? a : b));
        List<MobileAppMetrikaEvent> appMetrikaEvents = mobileAppGoalsMetrikaRepository
                .getEventsByMobileApps(mobileApps, true);
        goals.addAll(mapList(appMetrikaEvents, e -> appMetrikaEventToGoal(e, mobileAppsByKey)));

        return goals;
    }

    private Goal externalTrackerEventToGoal(MobileExternalTrackerEvent e, Map<Long, MobileApp> mobileAppMapsById) {
        return (Goal) new Goal()
                .withId(e.getId())
                .withName(defaultIfBlank(e.getCustomName(), e.getEventName().name()))
                .withMobileAppId(e.getMobileAppId())
                .withMobileAppName(mobileAppMapsById.get(e.getMobileAppId()).getName())
                .withMetrikaCounterGoalType(MetrikaCounterGoalType.ACTION)
                .withIsMobileGoal(true)
                .withAllowToUse(true);
    }

    private Goal appMetrikaEventToGoal(MobileAppMetrikaEvent e,
                                       Map<AppmetrikaUniqueKey, MobileApp> mobileAppsByKey) {
        AppmetrikaUniqueKey eKey = new AppmetrikaUniqueKey()
                .withAppMetrikaApplicationId(e.getAppMetrikaAppId())
                .withBundleId(e.getBundleId())
                .withStoreType(e.getStoreType());

        return (Goal) new Goal()
                .withId(e.getId())
                .withName(defaultIfBlank(e.getCustomName(), defaultAppmetrikaEventName(e)))
                .withMobileAppId(mobileAppsByKey.get(eKey).getId())
                .withMobileAppName(mobileAppsByKey.get(eKey).getName())
                .withMetrikaCounterGoalType(MetrikaCounterGoalType.ACTION)
                .withIsMobileGoal(true)
                .withAllowToUse(true);
    }

    private String defaultAppmetrikaEventName(MobileAppMetrikaEvent e) {
        return e.getEventType() == CLIENT ? e.getEventName()
                : AppmetrikaInternalEvent.fromTypeAndSubtype(e.getEventType(), e.getEventSubtype(), true).name();
    }
}
