package ru.yandex.direct.core.entity.mobilegoals.repository

import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.mobileapp.model.MobileGoalConversions
import ru.yandex.direct.grid.schema.yt.Tables
import ru.yandex.direct.ytcomponents.service.MobileEventsStatDynContextProvider
import ru.yandex.direct.ytwrapper.YtTableUtils
import ru.yandex.direct.ytwrapper.client.YtExecutionException
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL
import java.time.Instant
import java.time.temporal.ChronoUnit

@Repository
open class MobileGoalsStatisticRepository(
    private val dynContextProvider: MobileEventsStatDynContextProvider
) {
    companion object {
        private val MOBILE_EVENTS_STAT = Tables.DIRECTMOBILEEVENTSSTAT_BS.`as`("MobileEventsStat")
        private val GOALS_NUM = DSL.sum(MOBILE_EVENTS_STAT.GOALS_NUM).`as`("Conversions")
        private val GOAL_ID = YtTableUtils.aliased(MOBILE_EVENTS_STAT.GOAL_ID)
        private val logger = LoggerFactory.getLogger(MobileGoalsStatisticRepository::class.java)
    }

    fun getEventsStats(goalIds: Collection<Long>, daysNum: Int): List<MobileGoalConversions> {
        val timestampFrom: Long = Instant.now().minus(daysNum.toLong(), ChronoUnit.DAYS).epochSecond

        val query = YtDSL.ytContext()
            .select(GOAL_ID, GOALS_NUM)
            .from(MOBILE_EVENTS_STAT)
            .where(MOBILE_EVENTS_STAT.GOAL_ID.`in`(goalIds))
            .and(MOBILE_EVENTS_STAT.UPDATE_TIME.ge(timestampFrom))
            .groupBy(GOAL_ID)

        val rowset = try {
            dynContextProvider.getContext().executeSelect(query)
        } catch (ex: YtExecutionException) {
            logger.warn("Couldn't get goal conversion info", ex)
            return emptyList()
        }

        val goalIdColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, GOAL_ID)
        val goalsNumColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, GOALS_NUM)

        return rowset.rows.map {
            MobileGoalConversions(
                goalId = it.values[goalIdColIdx].longValue(),
                attributedConversions = it.values[goalsNumColIdx].longValue(),
                notAttributedConversions = 0
            )
        }
    }
}
