package ru.yandex.direct.core.entity.moderation;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.Region;

/**
 * Вспомагательный класс для проверок необходимости модерации для объектов Директа (группа объявлений, объявление и т.д.)
 */
@ParametersAreNonnullByDefault
public final class ModerationCheckUtils {
    private static final String FOREX_BANNER_FLAG = BannerFlags.FOREX.getKey();

    /**
     * Регионы при добавлении/удалении которых требуется перемодерация объявлений входящих в группу
     */
    private static final Set<Long> REMODERATABLE_REGION_IDS =
            ImmutableSet.of(Region.KAZAKHSTAN_REGION_ID);
    /**
     * Отображение специального региона в набор флагов объявлений при добавлении/удалении которого в группу объявлений
     * требуется перемодерация всех объявлений с данным флагом. Наличие данного флага в объявлении требует особой лицензии,
     * поэтому и необходима модерация
     */
    private static final Multimap<Long, String> REMODERATABLE_REGION_IDS_TO_BANNER_FLAGS =
            ImmutableSetMultimap.of(Region.RUSSIA_REGION_ID, FOREX_BANNER_FLAG);

    private ModerationCheckUtils() {
    }

    /**
     * Проверить необходима ли перемодерация объявлений группы при заданном изменении geo (Cм. ModerationChecks.pm::check_moderate_region)
     *
     * @param geoTree   Используемое геодерево
     * @param oldGeoIds Список исходных регионов
     * @param newGeoIds Список новых регионов
     * @return true если необходима
     */
    public static boolean isRemoderationRequiredByGeoChange(
            GeoTree geoTree,
            Collection<Long> oldGeoIds,
            Collection<Long> newGeoIds) {
        Objects.requireNonNull(geoTree, "geoTree");
        Objects.requireNonNull(oldGeoIds, "oldGeoIds");
        Objects.requireNonNull(newGeoIds, "newGeoIds");

        // TODO: Кажется текущий вариант можно упростить если просто проверять, что изменилась страна и поэтому
        // скорее всего потребуется новая модерация и это кажется будет более правильным
        return REMODERATABLE_REGION_IDS.stream()
                .map(Collections::singleton)
                .anyMatch(
                        // Проверяем что в списке geo появилась одна из специальных регионов наличие которого требует модерации
                        regionIdSet -> geoTree.isRegionsIncludedIn(newGeoIds, regionIdSet)
                                // но при этом ее не было раньше
                                && !geoTree.isRegionsIncludedIn(oldGeoIds, regionIdSet));
    }

    /**
     * Проверить необходима ли перемодерация  объявлений для которых требуется перемодерация, с учетом изменения geo
     * группы объявлений (Cм. ModerationChecks.pm::check_moderate_banner_for_regions)
     *
     * @param geoTree     Используемое геодерево
     * @param oldGeoIds   Список исходных регионов
     * @param newGeoIds   Список новых регионов
     * @param bannerFlags Флаги объявления
     * @return true если необходима
     */
    public static boolean isRemoderationRequiredForBannerByGeoChange(
            GeoTree geoTree, @Nullable BannerFlags bannerFlags, Set<Long> oldGeoIds, Set<Long> newGeoIds) {
        Map<String, String> bannerFlagsAsMap = bannerFlags != null ? bannerFlags.getFlags() : Collections.emptyMap();
        return EntryStream.of(REMODERATABLE_REGION_IDS_TO_BANNER_FLAGS.asMap())
                .mapKeys(Collections::singleton)
                .anyMatch(
                        e -> {
                            // Проверяем что объявление содержит хотя бы один флаг наличие которого требует модерации
                            boolean bannerHasSpecialFlag =
                                    e.getValue().stream().anyMatch(bannerFlagsAsMap::containsKey);
                            if (!bannerHasSpecialFlag) {
                                return false;
                            }

                            // И новый список geo содержит данный специальный регион или один из его подрегионов
                            boolean newGeoIncludedInSpecialRegion = geoTree.isAnyRegionOrSubRegionIncludedIn(
                                    e.getKey(), newGeoIds);
                            if (!newGeoIncludedInSpecialRegion) {
                                return false;
                            }

                            // Но при этом старый список geo не содержал данного специального региона или одного из его подрегионов
                            return !geoTree.isAnyRegionOrSubRegionIncludedIn(
                                    e.getKey(), oldGeoIds);
                        });
    }

    /**
     * Проверить необходима ли перемодерация мультибанера из-за изменения geo группы объявлений
     *
     * @param geoTree     Используемое геодерево
     * @param oldGeoIds   Список исходных регионов
     * @param newGeoIds   Список новых регионов
     * @return true если необходима
     */
    public static boolean isRemoderationRequiredForMulticardByGeoChange(
            GeoTree geoTree, Set<Long> oldGeoIds, Set<Long> newGeoIds) {
        var oldCountries = geoTree.getModerationCountries(oldGeoIds);
        var newCountries = geoTree.getModerationCountries(newGeoIds);
        return !oldCountries.equals(newCountries);
    }
}
