package ru.yandex.direct.core.entity.moderation;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;

import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.NORMAL;
import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;

@ParametersAreNonnullByDefault
@Component
public class ModerationOperationModeProvider {

    private final PpcProperty<Boolean> restrictedAdGroupTransportProperty;
    private final PpcProperty<Boolean> restrictedHtml5TransportProperty;
    private final PpcProperty<Boolean> restrictedCanvasTransportProperty;
    private final PpcProperty<Boolean> restrictedTextBannerTransportProperty;
    private final PpcProperty<Boolean> restrictedVcardAssetTransportProperty;
    private final PpcProperty<Boolean> restrictedVideoAdditionTransportProperty;
    private final PpcProperty<Boolean> restrictedCalloutTransportProperty;
    private final PpcProperty<Boolean> restrictedMobileContentTransportProperty;
    private final PpcProperty<Boolean> restrictedMobileContentIconTransportProperty;
    private final PpcProperty<Boolean> restrictedDynamicTransportProperty;

    private static final ThreadLocal<ModerationOperationMode> forced =
            ThreadLocal.withInitial(() -> null);

    private static final ThreadLocal<ModerationVersionMode> versionMode =
            ThreadLocal.withInitial(() -> ModerationVersionMode.NORMAL);


    @Autowired
    public ModerationOperationModeProvider(PpcPropertiesSupport ppcPropertiesSupport) {
        this.restrictedAdGroupTransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_ADGROUP_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
        this.restrictedHtml5TransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_HTML5_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
        this.restrictedCanvasTransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_CANVAS_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
        this.restrictedTextBannerTransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_TEXT_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
        this.restrictedVcardAssetTransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_VCARD_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
        this.restrictedVideoAdditionTransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_VIDEO_ADDITION_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
        this.restrictedCalloutTransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_CALLOUT_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
        this.restrictedMobileContentTransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_MOBILE_CONTENT_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
        this.restrictedMobileContentIconTransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_MOBILE_CONTENT_ICON_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
        this.restrictedDynamicTransportProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.RESTRICTED_DYNAMIC_TRANSPORT_NEW_MODERATION, Duration.ofMinutes(1));
    }

    public boolean isForcedRestrictedMode() {
        return forced.get() == RESTRICTED;
    }

    public ModerationOperationMode getMode(ModerationObjectType objectsType) {
        if (forced.get() != null) {
            return forced.get();
        }

        boolean restrictedTransportEnabled;

        // если пропертя еще не доехала до базы, работаем в полнофункциональном режиме
        switch (objectsType) {
            case ADGROUP:
                restrictedTransportEnabled = restrictedAdGroupTransportProperty.getOrDefault(false);
                break;
            case HTML5:
                restrictedTransportEnabled = restrictedHtml5TransportProperty.getOrDefault(false);
                break;
            case CANVAS:
                restrictedTransportEnabled = restrictedCanvasTransportProperty.getOrDefault(false);
                break;
            case TEXT_AD:
                restrictedTransportEnabled = restrictedTextBannerTransportProperty.getOrDefault(false);
                break;
            case BANNER_VCARD:
                restrictedTransportEnabled = restrictedVcardAssetTransportProperty.getOrDefault(false);
                break;
            case BANNER_VIDEO_ADDITION:
                restrictedTransportEnabled = restrictedVideoAdditionTransportProperty.getOrDefault(false);
                break;
            case CALLOUT:
                restrictedTransportEnabled = restrictedCalloutTransportProperty.getOrDefault(false);
                break;
            case MOBILE_CONTENT:
                restrictedTransportEnabled = restrictedMobileContentTransportProperty.getOrDefault(false);
                break;
            case MOBILE_CONTENT_ICON:
                restrictedTransportEnabled = restrictedMobileContentIconTransportProperty.getOrDefault(false);
                break;
            case DYNAMIC:
                restrictedTransportEnabled = restrictedDynamicTransportProperty.getOrDefault(false);
                break;
            default:
                restrictedTransportEnabled = false;
        }

        return restrictedTransportEnabled ? RESTRICTED : NORMAL;
    }

    public void forceMode(ModerationOperationMode mode) {
        forced.set(mode);
    }

    public void disableForcedMode() {
        forced.set(null);
    }

    public boolean isImmutableVersionMode() {
        return versionMode.get() == ModerationVersionMode.IMMUTABLE;
    }

    public void setImmutableVersionMode() {
        versionMode.set(ModerationVersionMode.IMMUTABLE);
    }

    public void disableImmutableVersionMode() {
        versionMode.set(ModerationVersionMode.NORMAL);
    }
}
