package ru.yandex.direct.core.entity.moderation.repository;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.DeleteConditionStep;
import org.jooq.InsertValuesStep1;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.addition.model.AdditionsModerateType;
import ru.yandex.direct.core.entity.addition.model.ModerateAdditions;
import ru.yandex.direct.dbschema.ppc.enums.ModEditType;
import ru.yandex.direct.dbschema.ppc.enums.ModObjectVersionObjType;
import ru.yandex.direct.dbschema.ppc.enums.ModReasonsType;
import ru.yandex.direct.dbschema.ppc.tables.records.ModExportCandidatesRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.ModObjectVersionRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.ModReasonsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.PostModerateRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.singleton;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.MODERATE_ADDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOD_EDIT;
import static ru.yandex.direct.dbschema.ppc.Tables.MOD_EXPORT_CANDIDATES;
import static ru.yandex.direct.dbschema.ppc.Tables.MOD_OBJECT_VERSION;
import static ru.yandex.direct.dbschema.ppc.Tables.MOD_REASONS;
import static ru.yandex.direct.dbschema.ppc.Tables.POST_MODERATE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class ModerationRepository {

    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<ModerateAdditions> moderateAdditionsMapper;

    @Autowired
    public ModerationRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.moderateAdditionsMapper = createModerateAdditionsMapper();
    }

    private static JooqMapperWithSupplier<ModerateAdditions> createModerateAdditionsMapper() {
        return JooqMapperWithSupplierBuilder.builder(ModerateAdditions::new)
                .map(property(ModerateAdditions.ID, MODERATE_ADDITIONS.ADDITIONS_ITEM_ID))
                .map(convertibleProperty(ModerateAdditions.MODERATE_TYPE, MODERATE_ADDITIONS.MODERATE_TYPE,
                        AdditionsModerateType::fromSource,
                        AdditionsModerateType::toSource))
                .build();
    }

    public void addPostModerate(DSLContext context, Set<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        InsertValuesStep1<PostModerateRecord, Long> insert = context.insertInto(POST_MODERATE, POST_MODERATE.BID);
        bannerIds.forEach(insert::values);
        insert.execute();
    }

    public void deleteBannerContactInfoVersion(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModObjectVersion(dsl, ModObjectVersionObjType.contactinfo, bannerIds);
    }

    public void deleteBannerDisplayHrefVersion(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModObjectVersion(dsl, ModObjectVersionObjType.display_href, bannerIds);
    }

    public void deleteBannerImageVersion(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModObjectVersion(dsl, ModObjectVersionObjType.image, bannerIds);
    }

    public void deleteBannerSitelinkSetVersion(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModObjectVersion(dsl, ModObjectVersionObjType.sitelinks_set, bannerIds);
    }

    public void deleteBannerVideoAdditionVersion(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModObjectVersion(dsl, ModObjectVersionObjType.video_addition, bannerIds);
    }

    public void deleteBannerTurboLandingVersion(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModObjectVersion(dsl, ModObjectVersionObjType.turbolanding, bannerIds);
    }

    /**
     * Удаляет записи о заданных группах из таблицы MOD_OBJECT_VERSION.
     *
     * @param shard      шард, в котором хранятся заданные группы
     * @param adGroupIds идентификаторы групп, записи о которых необходимо удалить
     */
    public void deleteAdGroupVersion(int shard, Collection<Long> adGroupIds) {
        deleteModObjectVersion(shard, ModObjectVersionObjType.phrases, adGroupIds);
    }

    public void deleteBannerContactInfoModReason(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModReason(dsl, ModReasonsType.contactinfo, bannerIds);
    }

    public void deleteBannerDisplayHrefModReason(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModReason(dsl, ModReasonsType.display_href, bannerIds);
    }

    public void deleteBannerImageModReason(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModReason(dsl, ModReasonsType.image, bannerIds);
    }

    public void deleteBannerSitelinkSetModReason(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModReason(dsl, ModReasonsType.sitelinks_set, bannerIds);
    }

    public void deleteBannerVideoAdditionModReason(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModReason(dsl, ModReasonsType.video_addition, bannerIds);
    }

    public void deleteBannerTurboLandingModReason(DSLContext dsl, Collection<Long> bannerIds) {
        deleteModReason(dsl, ModReasonsType.turbolanding, bannerIds);
    }

    /**
     * Удаляет записи о заданных группах из таблицы MOD_REASON.
     *
     * @param shard      шард, в котором хранятся заданные группы
     * @param adGroupIds идентификаторы групп, записи о которых необходимо удалить
     */
    public void deleteAdGroupModReason(int shard, Collection<Long> adGroupIds) {
        deleteModReason(shard, ModReasonsType.phrases, adGroupIds);
    }

    public void deleteBannerPageModReason(DSLContext dslContext, Collection<Long> moderateBannerPageIds) {
        deleteModReasonBatch(dslContext, singleton(ModReasonsType.banner_page), moderateBannerPageIds).execute();
    }

    public void deletePostModerate(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        deletePostModerate(dslContextProvider.ppc(shard).configuration(), bannerIds);
    }

    public void deletePostModerate(Configuration configuration, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        configuration.dsl()
                .deleteFrom(POST_MODERATE)
                .where(POST_MODERATE.BID.in(bannerIds))
                .execute();
    }

    public void deleteAutoModerate(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        deleteAutoModerate(dslContextProvider.ppc(shard).configuration(), bannerIds);
    }

    public void deleteAutoModerate(Configuration configuration, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        configuration.dsl()
                .deleteFrom(AUTO_MODERATE)
                .where(AUTO_MODERATE.BID.in(bannerIds))
                .execute();
    }

    public void deleteBannerModEdit(DSLContext dslContext, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        dslContext
                .deleteFrom(MOD_EDIT)
                .where(MOD_EDIT.ID.in(bannerIds)
                        .and(MOD_EDIT.TYPE.eq(ModEditType.banner)))
                .execute();
    }

    private void deleteModObjectVersion(int shard, ModObjectVersionObjType type, Collection<Long> bannerIds) {
        deleteModObjectVersion(dslContextProvider.ppc(shard), singleton(type), bannerIds);
    }

    private void deleteModObjectVersion(DSLContext dsl, ModObjectVersionObjType type, Collection<Long> bannerIds) {
        deleteModObjectVersion(dsl, singleton(type), bannerIds);
    }

    private void deleteModObjectVersion(DSLContext dsl, Collection<ModObjectVersionObjType> type,
                                        Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        deleteModObjectVersionBatch(dsl, type, bannerIds).execute();
    }

    public DeleteConditionStep<ModObjectVersionRecord> deleteModObjectVersionBatch(DSLContext context,
                                                                                   Collection<ModObjectVersionObjType> type, Collection<Long> bannerIds) {
        return context
                .deleteFrom(MOD_OBJECT_VERSION)
                .where(MOD_OBJECT_VERSION.OBJ_TYPE.in(type))
                .and(MOD_OBJECT_VERSION.OBJ_ID.in(bannerIds));
    }

    private void deleteModReason(int shard, ModReasonsType modReasonsType, Collection<Long> bannerIds) {
        deleteModReason(dslContextProvider.ppc(shard), singleton(modReasonsType), bannerIds);
    }

    private void deleteModReason(DSLContext dsl, ModReasonsType modReasonsType, Collection<Long> bannerIds) {
        deleteModReason(dsl, singleton(modReasonsType), bannerIds);
    }

    private void deleteModReason(DSLContext dsl, Collection<ModReasonsType> modReasonsType,
                                 Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        deleteModReasonBatch(dsl, modReasonsType, bannerIds).execute();
    }

    public DeleteConditionStep<ModReasonsRecord> deleteModReasonBatch(DSLContext context,
                                                                      Collection<ModReasonsType> modReasonsType,
                                                                      Collection<Long> bannerIds) {
        return context
                .delete(MOD_REASONS)
                .where(MOD_REASONS.ID.in(bannerIds))
                .and(MOD_REASONS.TYPE.in(modReasonsType));
    }

    public int addModerateAdditions(int shard, Collection<ModerateAdditions> moderateAdditions) {
        return new InsertHelper<>(dslContextProvider.ppc(shard), MODERATE_ADDITIONS)
                .addAll(moderateAdditionsMapper, moderateAdditions)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    public List<ModerateAdditions> getModerateAdditions(int shard, Collection<Long> additionsId) {
        return dslContextProvider.ppc(shard)
                .select(moderateAdditionsMapper.getFieldsToRead())
                .from(MODERATE_ADDITIONS)
                .where(MODERATE_ADDITIONS.ADDITIONS_ITEM_ID.in(additionsId))
                .fetch(moderateAdditionsMapper::fromDb);
    }

    public void insertIgnoreModExportCandidates(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return;
        }
        InsertValuesStep1<ModExportCandidatesRecord, Long> insert = dslContextProvider.ppc(shard)
                .insertInto(MOD_EXPORT_CANDIDATES, MOD_EXPORT_CANDIDATES.CID);
        campaignIds.forEach(insert::values);
        insert.onDuplicateKeyIgnore().execute();
    }

    public void deleteModExportCandidates(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard)
                .deleteFrom(MOD_EXPORT_CANDIDATES)
                .where(MOD_EXPORT_CANDIDATES.CID.in(campaignIds))
                .execute();
    }
}
