package ru.yandex.direct.core.entity.moderation.repository.bulk_update;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Predicate;

import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.TableField;

/**
 *
 * Класс нужен для накопления апдейтов к базе данных и массового их выполнения за один  запрос.
 * Это нужно, т.к мы обрабатываем  вердикты модерации  с помощью независимых операций, каждая из которых обрабатывает
 * одни и те же объекты, и в результате мы можем иметь число запросов в одну таблицу по числу операций.
 * Исторически не все операции были массовыми, что делало все еще хуже, их исправление приводило к дублирующемуся коду
 * массовых апдейтов.
 *
 * Вторая причина - перед выполнением запроса в базу хочется посмотреть и провалидировать что
 * все операции совместно не повредят статусы объектов.
 *
 *  setValueMerger - устанавливает метод для слияния значений (например разные операции хотят выставить разное значение одному полю)
 *  setValidator - устанавливает валидатор для данных перед запросом в базу
 *
 */

public class BulkUpdateHolder {
    private final Map<TableField<?, ?>, BulkUpdate<?, ?>> bulks = new HashMap<>();
    private BiFunction<Object, Object, Object> valueMerger = this::defaultValueMerger;
    private Predicate<BulkUpdate.RowChanges<? extends Record>> validator = e -> true;

    public BulkUpdateHolder setValueMerger(BiFunction<Object, Object, Object> valueMerger) {
        this.valueMerger = valueMerger;
        return this;
    }

    public BulkUpdateHolder setValidator(Predicate<BulkUpdate.RowChanges<? extends Record>> validator) {
        this.validator = validator;
        return this;
    }

    private Object defaultValueMerger(Object a, Object b) {
        if (a != null && a.equals(b)) {
            return a;
        }

        throw new IllegalStateException("Trying to change already set value for field " + a);
    }

    @SuppressWarnings("unchecked")
    public <T extends Record, H extends Number> BulkUpdate<T, H> get(TableField<T, H> keyField) {
        return (BulkUpdate<T, H>) bulks.computeIfAbsent(keyField, t -> new BulkUpdate<>(keyField, valueMerger,
                validator));
    }

    public void execute(Configuration configuration) {
        bulks.values().stream()
                .sorted(Comparator.comparing(BulkUpdate::getTableName))
                .forEach(e -> e.execute(configuration));
        //Объект инициализируется один раз, но может использоваться несколько раз. Поэтому делаем clear.
        bulks.clear();
    }
}
