package ru.yandex.direct.core.entity.moderation.repository.receiving;

import java.util.Collection;
import java.util.List;

import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Record1;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.core.entity.moderation.repository.sending.TransportStatusAdapter;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesCreativeType;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.PerfCreativesRecord;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.BannersPerformance.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.tables.BannerstorageCreativesModerationVersions.BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.tables.PerfCreatives.PERF_CREATIVES;
import static ru.yandex.direct.dbutil.SqlUtils.PRIMARY;

@Repository
public class BannerstorageCreativesModerationReceivingRepository extends
        ModerationReceivingRepositorySingleKey<PerfCreativesRecord, PerfCreativesStatusmoderate> {

    public static final String BANNERSTORAGE_CREATIVE_MODERATED_BY_DIRECT_MARKER = "{\"moderated_by_direct\": true}";

    public BannerstorageCreativesModerationReceivingRepository() {
        super(
                PERF_CREATIVES.CREATIVE_ID, // если ключ менять, то надо ниже убрать forceIndex(PRIMARY)
                BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS.VERSION,
                PERF_CREATIVES.STATUS_MODERATE);
    }

    @Override
    PerfCreativesStatusmoderate transportStatusMap(TransportStatus transportStatus) {
        return TransportStatusAdapter.toPerfCreativesStatusmoderate(transportStatus);
    }

    @Override
    SelectJoinStep<Record1<Long>> getSelectionStep(Configuration configuration) {
        return DSL.using(configuration)
                .selectDistinct(PERF_CREATIVES.CREATIVE_ID)
                .from(PERF_CREATIVES.forceIndex(PRIMARY))
                .join(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .leftJoin(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS)
                .on(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS.BID.eq(BANNERS_PERFORMANCE.BID));
    }

    /**
     * Устанавливает статус модерации для указанных креативов
     * Для креативов из bannerstorage дополнительно устанавливается маркер того, что креатив был
     * отмодерирован в Модерации Директа (чтобы можно было их отличать от тех,
     * которые были отмодерированы в bannerstorage)
     */
    public void setStatusModerate(DSLContext dslContext,
                                  Collection<Long> creativeIds,
                                  PerfCreativesStatusmoderate statusModerate) {
        if (creativeIds.isEmpty()) {
            return;
        }
        dslContext
                .update(PERF_CREATIVES)
                .set(PERF_CREATIVES.STATUS_MODERATE, statusModerate)
                .set(PERF_CREATIVES.ADDITIONAL_DATA,
                        // Для креативов из bannerstorage ставим маркер, для остальных оставляем как есть
                        DSL.iif(PERF_CREATIVES.CREATIVE_TYPE.in(
                                PerfCreativesCreativeType.performance,
                                PerfCreativesCreativeType.bannerstorage
                        ), BANNERSTORAGE_CREATIVE_MODERATED_BY_DIRECT_MARKER, PERF_CREATIVES.ADDITIONAL_DATA))
                .where(PERF_CREATIVES.CREATIVE_ID.in(creativeIds))
                .execute();
    }

    /**
     * Возвращает номера всех баннеров, связанных с указанными креативами
     */
    public List<Long> getBannersLinkedToCreatives(DSLContext dslContext,
                                                  Collection<Long> creativeIds) {
        if (creativeIds.isEmpty()) {
            return emptyList();
        }
        return dslContext
                .select(BANNERS_PERFORMANCE.BID)
                .from(BANNERS_PERFORMANCE)
                .join(BANNERS).on(BANNERS_PERFORMANCE.BID.eq(BANNERS.BID))
                .where(
                        BANNERS_PERFORMANCE.CREATIVE_ID.in(creativeIds)
                                .and(BANNERS.STATUS_MODERATE.ne(BannersStatusmoderate.New))
                )
                .fetch(BANNERS_PERFORMANCE.BID);
    }
}
