package ru.yandex.direct.core.entity.moderation.repository.receiving;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.SelectJoinStep;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.impl.DSL;

import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.core.entity.moderation.service.receiving.BaseModerationReceivingService;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.No;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Ready;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Sending;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Sent;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Yes;
import static ru.yandex.direct.core.entity.moderation.service.receiving.BaseModerationReceivingService.ModeratedObjectKeyWithVersion.ANY_VERSION;
import static ru.yandex.direct.dbutil.SqlUtils.PRIMARY;

public abstract class ModerationReceivingRepository<H extends Record, T extends Enum<T>, U> {
    private final List<TableField<H, Long>> keyFields;
    private final TableField<?, Long> versionField;
    private final TableField<H, T> statusModerateField;

    private final static EnumSet<TransportStatus> POSSIBLE_STATUSES_WHEN_IGNORE_VERSION =
            EnumSet.of(Yes, No, Sending, Sent, Ready);
    private final static EnumSet<TransportStatus> POSSIBLE_STATUSES_FOR_CURRENT_VERSION = EnumSet.of(Yes, No, Sending,
            Sent);

    protected ModerationReceivingRepository(TableField<H, Long> keyField, TableField<?, Long> versionField,
                                            TableField<H, T> statusModerateField) {
        this(List.of(keyField), versionField, statusModerateField);
    }

    protected ModerationReceivingRepository(List<TableField<H, Long>> keyFields, TableField<?, Long> versionField,
                                            TableField<H, T> statusModerateField) {
        checkArgument(!keyFields.isEmpty());
        this.keyFields = keyFields;
        this.versionField = versionField;
        this.statusModerateField = statusModerateField;
    }

    protected List<TableField<H, Long>> getKeyFields() {
        return keyFields;
    }

    protected TableField<?, Long> getVersionField() {
        return versionField;
    }

    SelectJoinStep<? extends Record> getSelectionStep(Configuration configuration) {
        var versionField = getVersionField();

        Table<H> table = keyFields.get(0).getTable();
        if (keyFields.equals(table.getPrimaryKey().getFields())) {
            // y MySQL иногда сходит с ума построение плана запроса, помогаем ему не ошибиться
            table = table.forceIndex(PRIMARY);
        }
        SelectJoinStep<Record> selectStep = DSL.using(configuration)
                .select(keyFields)
                .from(table);
        if (keyFields.get(0).getTable().equals(versionField.getTable())) {
            return selectStep;
        }
        return selectStep.leftJoin(versionField.getTable())
                .using(keyFields);
    }

    abstract T transportStatusMap(TransportStatus transportStatus);

    protected abstract Condition getWhereConditionForObject(BaseModerationReceivingService.ModeratedObjectKeyWithVersion<U> objectKeyWithVersion);

    protected abstract U mapRecordToKey(Record record);

    public List<U> selectByKeysAndExportVersionsWithLock(Configuration configuration,
                                                         Collection<BaseModerationReceivingService.ModeratedObjectKeyWithVersion<U>> keysAndVersions) {
        if (keysAndVersions.isEmpty()) {
            return List.of();
        }

        var selectStep = getSelectionStep(configuration);
        var condition = makeConditionForObjects(keysAndVersions);

        return selectStep.where(condition).forUpdate().fetch(this::mapRecordToKey);
    }

    Condition makeConditionForObjects(
            Collection<BaseModerationReceivingService.ModeratedObjectKeyWithVersion<U>> keysAndVersions) {

        List<Condition> conditions = new ArrayList<>();

        Map<Set<? super T>, List<Condition>> byStatuses = new HashMap<>();

        for (var object : keysAndVersions) {
            Set<? super T> statusesConverted =
                    getAllowedStatusesForObject(object)
                            .stream()
                            .map(this::transportStatusMap)
                            .collect(Collectors.toSet());

            List<Condition> conds = byStatuses.computeIfAbsent(statusesConverted, s -> new ArrayList<>());

            if (object.getVersionId() == ANY_VERSION) {
                conds.add(getWhereConditionForObject(object));
            } else {
                conds.add(getWhereConditionForObject(object)
                        .and(DSL.or(versionField.eq(object.getVersionId()), versionField.isNull()))
                );
            }
        }

        for (var status : byStatuses.keySet()) {
            conditions.add(DSL.and(statusModerateField.in(status), DSL.or(byStatuses.get(status))));
        }

        return DSL.or(conditions);
    }

    EnumSet<TransportStatus> getAllowedStatusesForObject(BaseModerationReceivingService.ModeratedObjectKeyWithVersion<U> objectKeyWithVersion) {
        if (objectKeyWithVersion.getVersionId() == ANY_VERSION) {
            return POSSIBLE_STATUSES_WHEN_IGNORE_VERSION;
        } else {
            return POSSIBLE_STATUSES_FOR_CURRENT_VERSION;
        }
    }

}
