package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.InsertSetMoreStep;
import org.jooq.Record1;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithModerationInfo;
import ru.yandex.direct.core.entity.keyword.model.KeywordModeration;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupsModerationVersionsRecord;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.repository.sending.TransportStatusAdapter.toAdGroupStatusModerate;
import static ru.yandex.direct.core.entity.moderation.repository.sending.TransportStatusAdapter.toKeywordsStatusModerate;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.ADGROUP;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class AdGroupSendingRepository implements ModerationSendingRepository<Long, AdGroupWithModerationInfo> {

    private final KeywordRepository keywordRepository;
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final JooqReaderWithSupplier<AdGroupWithModerationInfo> jooqReader;

    @Autowired
    public AdGroupSendingRepository(KeywordRepository keywordRepository,
                                    ModerationOperationModeProvider moderationOperationModeProvider) {
        this.keywordRepository = keywordRepository;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        this.jooqReader = createJooqReader();
    }

    private JooqReaderWithSupplier<AdGroupWithModerationInfo> createJooqReader() {
        return JooqReaderWithSupplierBuilder.builder(AdGroupWithModerationInfo::new)
                .readProperty(AdGroupWithModerationInfo.ID, fromField(PHRASES.PID))
                .readProperty(AdGroupWithModerationInfo.CAMPAIGN_ID, fromField(PHRASES.CID))
                .readProperty(AdGroupWithModerationInfo.CLIENT_ID, fromField(CAMPAIGNS.CLIENT_ID))
                .readProperty(AdGroupWithModerationInfo.UID, fromField(CAMPAIGNS.UID))
                .readProperty(AdGroupWithModerationInfo.CLIENT_FLAGS,
                        fromField(CLIENTS_OPTIONS.CLIENT_FLAGS).by(RepositoryUtils::setFromDb))
                .readProperty(AdGroupWithModerationInfo.TRANSPORT_STATUS,
                        fromField(PHRASES.STATUS_MODERATE).by(TransportStatusAdapter::fromDb))
                .readProperty(AdGroupWithModerationInfo.VERSION, fromField(ADGROUPS_MODERATION_VERSIONS.VERSION))
                .build();
    }

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        return DSL.using(configuration)
                .select(DSL.count().cast(Long.class))
                .from(PHRASES)
                .where(PHRASES.STATUS_MODERATE.eq(PhrasesStatusmoderate.Ready))
                .fetchOne(Record1::value1);
    }

    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration configuration) {
        return DSL.using(configuration)
                .select(PHRASES.PID)
                .from(PHRASES)
                .where(PHRASES.PID.in(keys))
                .and(getStatusModerateCondition())
                .forUpdate()
                .fetch(PHRASES.PID);
    }

    private Condition getStatusModerateCondition() {
        if (moderationOperationModeProvider.getMode(ADGROUP) == RESTRICTED) {
            return DSL.trueCondition();
        }
        return PHRASES.STATUS_MODERATE.in(PhrasesStatusmoderate.Ready, PhrasesStatusmoderate.Sending);
    }

    @Override
    public List<AdGroupWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                   Configuration configuration) {
        List<AdGroupWithModerationInfo> adGroupWithModerationInfoList = DSL.using(configuration)
                .select(jooqReader.getFieldsToRead())
                .from(PHRASES)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .leftJoin(ADGROUPS_MODERATION_VERSIONS).on(ADGROUPS_MODERATION_VERSIONS.PID.eq(PHRASES.PID))
                .where(PHRASES.PID.in(lockedKeys))
                .fetch(jooqReader::fromDb);

        Map<Long, List<KeywordModeration>> adGroupIdToKeywordText =
                keywordRepository.getKeywordModerationsByAdGroupIds(configuration, null, lockedKeys);

        adGroupWithModerationInfoList.forEach(adGroup -> {
            List<KeywordModeration> keywords = nvl(adGroupIdToKeywordText.get(adGroup.getId()), emptyList());
            adGroup.setKeywords(keywords);
        });

        return adGroupWithModerationInfoList;
    }

    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<AdGroupWithModerationInfo> adGroups) {
        if (moderationOperationModeProvider.getMode(ADGROUP) == RESTRICTED) {
            return;
        }

        List<Long> adGroupIds = mapList(adGroups, AdGroupWithModerationInfo::getId);

        DSL.using(config)
                .update(PHRASES)
                .set(PHRASES.STATUS_MODERATE, toAdGroupStatusModerate(to))
                .where(PHRASES.PID.in(adGroupIds))
                .and(PHRASES.STATUS_MODERATE.eq(toAdGroupStatusModerate(from)))
                .execute();
    }

    public void updateKeywordsStatusModerateByAdGroupIds(Configuration config, TransportStatus to,
                                                         Collection<Long> adGroupIds) {
        if (moderationOperationModeProvider.getMode(ADGROUP) == RESTRICTED) {
            return;
        }

        DSL.using(config)
                .update(BIDS)
                .set(BIDS.STATUS_MODERATE, toKeywordsStatusModerate(to))
                .where(BIDS.PID.in(adGroupIds))
                .execute();
    }

    @Override
    public void setModerationVersions(Configuration configuration,
                                      List<Pair<AdGroupWithModerationInfo, Long>> inserts) {
        if (inserts.isEmpty()) {
            return;
        }

        var insertSetStep = DSL.using(configuration)
                .insertInto(ADGROUPS_MODERATION_VERSIONS);
        InsertSetMoreStep<AdgroupsModerationVersionsRecord> insertSetMoreStep = null;
        for (Pair<AdGroupWithModerationInfo, Long> info : inserts) {
            var record = new AdgroupsModerationVersionsRecord();
            record.setPid(info.getLeft().getId());
            record.setVersion(info.getRight());

            if (insertSetMoreStep != null) {
                insertSetStep = insertSetMoreStep.newRecord();
            }

            insertSetMoreStep = insertSetStep.set(record);
        }

        insertSetMoreStep
                .onDuplicateKeyUpdate()
                .set(ADGROUPS_MODERATION_VERSIONS.VERSION,
                        MySQLDSL.values(ADGROUPS_MODERATION_VERSIONS.VERSION))
                .set(ADGROUPS_MODERATION_VERSIONS.CREATE_TIME,
                        MySQLDSL.currentLocalDateTime())
                .execute();
    }

    public void updateStatusmoderateAndStatusPostmoderate(Configuration config,
                                                          TransportStatus from,
                                                          TransportStatus to,
                                                          PhrasesStatuspostmoderate statusPostmoderate,
                                                          Collection<AdGroupWithModerationInfo> adGroups) {
        if (moderationOperationModeProvider.getMode(ADGROUP) == RESTRICTED) {
            return;
        }

        List<Long> adGroupIds = mapList(adGroups, AdGroupWithModerationInfo::getId);

        DSL.using(config)
                .update(PHRASES)
                .set(PHRASES.STATUS_MODERATE, toAdGroupStatusModerate(to))
                .set(PHRASES.STATUS_POST_MODERATE, statusPostmoderate)
                .where(PHRASES.PID.in(adGroupIds))
                .and(PHRASES.STATUS_MODERATE.eq(toAdGroupStatusModerate(from)))
                .execute();
    }
}
