package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;

import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.ModerationableWithBid;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public abstract class AssetModerationRepository<T extends Record, S, V extends Record, E extends ModerationableWithBid>
        implements ModerationSendingRepository<Long, E> {

    private final AssetModerationRepositoryParams<T, S, V> params;
    private final ModerationOperationModeProvider moderationOperationModeProvider;

    public AssetModerationRepository(AssetModerationRepositoryParams<T, S, V> params,
                                     ModerationOperationModeProvider moderationOperationModeProvider) {
        this.params = params;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
    }

    protected ModerationOperationModeProvider getModerationOperationModeProvider() {
        return moderationOperationModeProvider;
    }

    protected AssetModerationRepositoryParams<T, S, V> getParams() {
        return params;
    }

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        S ready = params.getTransportStatusConverter().apply(TransportStatus.Ready);
        return configuration.dsl()
                .select(DSL.count().cast(Long.class))
                .from(params.getTable())
                .where(params.getStatusModerateField().eq(ready))
                .fetchOne(Record1::value1);
    }

    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration config) {
        return DSL.using(config)
                .select(params.getIdField())
                .from(params.getTable())
                .where(params.getIdField().in(keys))
                .and(getStatusModerateCondition())
                .forUpdate()
                .fetch(r -> r.get(params.getIdField()));
    }

    @SuppressWarnings("unchecked")
    protected Condition getStatusModerateCondition() {
        // в ограниченном режиме не фильтруем по статусам модерации, подходят любые
        if (moderationOperationModeProvider.isForcedRestrictedMode()) {
            return DSL.trueCondition();
        }

        S ready = params.getTransportStatusConverter().apply(TransportStatus.Ready);
        S sending = params.getTransportStatusConverter().apply(TransportStatus.Sending);
        return params.getStatusModerateField().in(ready, sending);
    }

    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<E> loadedObjects) {
        // в ограниченном режиме не обновляем статусы модерации
        if (moderationOperationModeProvider.isForcedRestrictedMode()) {
            return;
        }

        S fromStatusmoderate = params.getTransportStatusConverter().apply(from);
        S toStatusmoderate = params.getTransportStatusConverter().apply(to);
        List<Long> ids = mapList(loadedObjects, E::getBid);

        DSL.using(config)
                .update(params.getTable())
                .set(params.getStatusModerateField(), toStatusmoderate)
                .where(
                        params.getStatusModerateField().eq(fromStatusmoderate),
                        params.getIdField().in(ids))
                .execute();
    }

    @Override
    public void setModerationVersions(Configuration configuration, List<Pair<E, Long>> inserts) {
        if (inserts.isEmpty()) {
            return;
        }

        InsertHelper<V> insertHelper = new InsertHelper<>(DSL.using(configuration), params.getVersionsTable());
        inserts.forEach(insert -> {
            insertHelper.set(params.getVersionsTableIdField(), insert.getLeft().getBid());
            insertHelper.set(params.getVersionField(), insert.getRight());
            insertHelper.newRecord();
        });

        insertHelper
                .onDuplicateKeyUpdate()
                .set(params.getVersionField(), MySQLDSL.values(params.getVersionField()))
                .set(params.getCreateTimeField(), MySQLDSL.currentLocalDateTime())
                .execute();
    }
}
