package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.RecordMapper;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerButtonWithModerationInfo;
import ru.yandex.direct.core.entity.hrefparams.service.HrefWithParamsBuildingService;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.dbschema.ppc.enums.BannerButtonsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerButtonsModerationVersionsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerButtonsRecord;

import static java.util.function.Function.identity;
import static ru.yandex.direct.common.util.RepositoryUtils.setFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_BUTTONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_BUTTONS_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.GROUP_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

@Repository
public class BannerButtonsSendingRepository
        extends AssetModerationRepository<BannerButtonsRecord, BannerButtonsStatusmoderate,
        BannerButtonsModerationVersionsRecord, BannerButtonWithModerationInfo> {

    public BannerButtonsSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(createRepositoryParams(), moderationOperationModeProvider);
    }

    private static AssetModerationRepositoryParams<BannerButtonsRecord,
            BannerButtonsStatusmoderate, BannerButtonsModerationVersionsRecord> createRepositoryParams() {
        return AssetModerationRepositoryParams.builder()
                .withTable(BANNER_BUTTONS)
                .withIdField(BANNER_BUTTONS.BID)
                .withStatusModerateField(BANNER_BUTTONS.STATUS_MODERATE)
                .withTransportStatusConverter(TransportStatusAdapter::toBannerButtonsStatusmoderate)
                .withVersionsTable(BANNER_BUTTONS_MODERATION_VERSIONS)
                .withVersionsTableIdField(BANNER_BUTTONS_MODERATION_VERSIONS.BID)
                .withVersionField(BANNER_BUTTONS_MODERATION_VERSIONS.VERSION)
                .withCreateTimeField(BANNER_BUTTONS_MODERATION_VERSIONS.CREATE_TIME)
                .build();
    }

    @Override
    public List<BannerButtonWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                        Configuration config) {
        return DSL.using(config)
                .select(BANNER_BUTTONS.BID,
                        BANNERS.PID,
                        BANNERS.CID,
                        CAMPAIGNS.CLIENT_ID,
                        CAMPAIGNS.UID,
                        BANNER_BUTTONS.HREF,
                        BANNER_BUTTONS.CAPTION,
                        BANNER_BUTTONS_MODERATION_VERSIONS.VERSION,
                        CAMP_OPTIONS.HREF_PARAMS,
                        GROUP_PARAMS.HREF_PARAMS,
                        PRE_MODERATE_BANNERS.BID,
                        AUTO_MODERATE.BID,
                        BANNER_BUTTONS.STATUS_MODERATE,
                        CLIENTS_OPTIONS.CLIENT_FLAGS)
                .from(BANNER_BUTTONS)
                .join(BANNERS).on(BANNER_BUTTONS.BID.eq(BANNERS.BID))
                .join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(CLIENTS_OPTIONS).on(CAMPAIGNS.CLIENT_ID.eq(CLIENTS_OPTIONS.CLIENT_ID))
                .leftJoin(BANNER_BUTTONS_MODERATION_VERSIONS).on(BANNER_BUTTONS_MODERATION_VERSIONS.BID.eq(BANNER_BUTTONS.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(BANNER_BUTTONS.BID.eq(PRE_MODERATE_BANNERS.BID))
                .leftJoin(AUTO_MODERATE).on(BANNER_BUTTONS.BID.eq(AUTO_MODERATE.BID))
                .leftJoin(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .leftJoin(GROUP_PARAMS).on(BANNERS.PID.eq(GROUP_PARAMS.PID))
                .where(BANNER_BUTTONS.BID.in(lockedKeys))
                .fetch(new BannerButtonRecordMapper());
    }


    private static class BannerButtonRecordMapper implements RecordMapper<Record, BannerButtonWithModerationInfo> {
        @Override
        public BannerButtonWithModerationInfo map(Record record) {
            return new BannerButtonWithModerationInfo()
                    .withId(record.get(BANNER_BUTTONS.BID))
                    .withAdGroupId(record.get(BANNERS.PID))
                    .withBid(record.get(BANNER_BUTTONS.BID))
                    .withCampaignId(record.get(BANNERS.CID))
                    .withClientId(record.get(CAMPAIGNS.CLIENT_ID))
                    .withUid(record.get(CAMPAIGNS.UID))
                    .withHref(buildButtonHref(record))
                    .withCaption(record.get(BANNER_BUTTONS.CAPTION))
                    .withBidAutoModerate(record.get(AUTO_MODERATE.BID))
                    .withBidReModerate(record.get(PRE_MODERATE_BANNERS.BID))
                    .withTransportStatus(TransportStatusAdapter.fromDb(record.get(BANNER_BUTTONS.STATUS_MODERATE)))
                    .withVersion(record.get(BANNER_BUTTONS_MODERATION_VERSIONS.VERSION))
                    .withClientFlags(setFromDb(record.get(CLIENTS_OPTIONS.CLIENT_FLAGS), identity()));
        }
    }

    private static String buildButtonHref(Record record) {
        return HrefWithParamsBuildingService.buildHrefWithParams(record.get(BANNER_BUTTONS.HREF),
                record.get(GROUP_PARAMS.HREF_PARAMS), record.get(CAMP_OPTIONS.HREF_PARAMS));
    }
}
