package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.RecordMapper;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerLogoWithModerationInfo;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.dbschema.ppc.enums.BannerLogosStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerLogosModerationVersionsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerLogosRecord;

import static java.util.function.Function.identity;
import static ru.yandex.direct.common.util.RepositoryUtils.setFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES_FORMATS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_LOGOS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_LOGOS_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

@Repository
public class BannerLogoSendingRepository
        extends AssetModerationRepository<BannerLogosRecord, BannerLogosStatusmoderate,
        BannerLogosModerationVersionsRecord, BannerLogoWithModerationInfo> {

    public BannerLogoSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(createRepositoryParams(), moderationOperationModeProvider);
    }

    private static AssetModerationRepositoryParams<BannerLogosRecord,
            BannerLogosStatusmoderate, BannerLogosModerationVersionsRecord> createRepositoryParams() {
        return AssetModerationRepositoryParams.builder()
                .withTable(BANNER_LOGOS)
                .withIdField(BANNER_LOGOS.BID)
                .withStatusModerateField(BANNER_LOGOS.STATUS_MODERATE)
                .withTransportStatusConverter(TransportStatusAdapter::toBannerLogosStatusmoderate)
                .withVersionsTable(BANNER_LOGOS_MODERATION_VERSIONS)
                .withVersionsTableIdField(BANNER_LOGOS_MODERATION_VERSIONS.BID)
                .withVersionField(BANNER_LOGOS_MODERATION_VERSIONS.VERSION)
                .withCreateTimeField(BANNER_LOGOS_MODERATION_VERSIONS.CREATE_TIME)
                .build();
    }

    @Override
    public List<BannerLogoWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                      Configuration config) {
        return DSL.using(config)
                .select(BANNER_LOGOS.BID,
                        BANNERS.PID,
                        BANNERS.CID,
                        CAMPAIGNS.CLIENT_ID,
                        CAMPAIGNS.UID,
                        BANNER_LOGOS.IMAGE_HASH,
                        BANNER_IMAGES_FORMATS.AVATARS_HOST, BANNER_IMAGES_FORMATS.MDS_GROUP_ID,
                        BANNER_IMAGES_FORMATS.NAMESPACE,
                        BANNER_LOGOS_MODERATION_VERSIONS.VERSION,
                        PRE_MODERATE_BANNERS.BID,
                        AUTO_MODERATE.BID,
                        BANNER_LOGOS.STATUS_MODERATE,
                        CLIENTS_OPTIONS.CLIENT_FLAGS)
                .from(BANNER_LOGOS)
                .join(BANNERS).on(BANNER_LOGOS.BID.eq(BANNERS.BID))
                .join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(BANNER_IMAGES_FORMATS).on(BANNER_LOGOS.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH))
                .leftJoin(CLIENTS_OPTIONS).on(CAMPAIGNS.CLIENT_ID.eq(CLIENTS_OPTIONS.CLIENT_ID))
                .leftJoin(BANNER_LOGOS_MODERATION_VERSIONS).on(BANNER_LOGOS_MODERATION_VERSIONS.BID.eq(BANNER_LOGOS.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(BANNER_LOGOS.BID.eq(PRE_MODERATE_BANNERS.BID))
                .leftJoin(AUTO_MODERATE).on(BANNER_LOGOS.BID.eq(AUTO_MODERATE.BID))
                .where(BANNER_LOGOS.BID.in(lockedKeys))
                .fetch(new BannerLogoRecordMapper());
    }

    private static class BannerLogoRecordMapper implements RecordMapper<Record, BannerLogoWithModerationInfo> {
        @Override
        public BannerLogoWithModerationInfo map(Record record) {
            BannerLogoWithModerationInfo bannerLogoWithModerationInfo = new BannerLogoWithModerationInfo()
                    .withAdGroupId(record.get(BANNERS.PID))
                    .withBid(record.get(BANNER_LOGOS.BID))
                    .withCampaignId(record.get(BANNERS.CID))
                    .withClientId(record.get(CAMPAIGNS.CLIENT_ID))
                    .withUid(record.get(CAMPAIGNS.UID))
                    .withImageHash(record.get(BANNER_LOGOS.IMAGE_HASH))
                    .withBidAutoModerate(record.get(AUTO_MODERATE.BID))
                    .withBidReModerate(record.get(PRE_MODERATE_BANNERS.BID))
                    .withTransportStatus(TransportStatusAdapter.fromDb(record.get(BANNER_LOGOS.STATUS_MODERATE)))
                    .withVersion(record.get(BANNER_LOGOS_MODERATION_VERSIONS.VERSION))
                    .withClientFlags(setFromDb(record.get(CLIENTS_OPTIONS.CLIENT_FLAGS), identity()));

            if (record.get(BANNER_IMAGES_FORMATS.MDS_GROUP_ID) != null) {
                bannerLogoWithModerationInfo.withAvatarsHost(AvatarHost.fromSource(record.get(BANNER_IMAGES_FORMATS.AVATARS_HOST)))
                        .withNamespace(BannerImageFormatNamespace.fromSource(record.get(BANNER_IMAGES_FORMATS.NAMESPACE)))
                        .withMdsGroupId(record.get(BANNER_IMAGES_FORMATS.MDS_GROUP_ID).intValue());

            }

            return bannerLogoWithModerationInfo;
        }
    }

}
