package ru.yandex.direct.core.entity.moderation.repository.sending

import org.jooq.Configuration
import org.jooq.Record
import org.jooq.TransactionalRunnable
import org.jooq.impl.DSL.count
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.util.RepositoryUtils
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings
import ru.yandex.direct.core.entity.banner.model.BannerMulticardSetWithModerationInfo
import ru.yandex.direct.core.entity.banner.model.BannerMulticardWithModerationInfo
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.Tables.*
import ru.yandex.direct.dbschema.ppc.enums.BannerMulticardSetsStatusmoderate
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesCreativeType
import ru.yandex.direct.dbschema.ppc.tables.AutoModerate.AUTO_MODERATE
import ru.yandex.direct.dbschema.ppc.tables.BannerMulticardSets.BANNER_MULTICARD_SETS
import ru.yandex.direct.dbschema.ppc.tables.BannerMulticardSetsModerationVersions.BANNER_MULTICARD_SETS_MODERATION_VERSIONS
import ru.yandex.direct.dbschema.ppc.tables.BannerMulticards.BANNER_MULTICARDS
import ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS
import ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.tables.ClientsOptions.CLIENTS_OPTIONS
import ru.yandex.direct.dbschema.ppc.tables.PreModerateBanners.PRE_MODERATE_BANNERS
import ru.yandex.direct.dbschema.ppc.tables.records.BannerMulticardSetsModerationVersionsRecord
import ru.yandex.direct.dbschema.ppc.tables.records.BannerMulticardSetsRecord
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property
import ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField

@Repository
class BannerMulticardSetSendingRepository @Autowired constructor(
    private val dslContextProvider: DslContextProvider,
    moderationOperationModeProvider: ModerationOperationModeProvider
) : AssetModerationRepository<BannerMulticardSetsRecord, BannerMulticardSetsStatusmoderate,
    BannerMulticardSetsModerationVersionsRecord, BannerMulticardSetWithModerationInfo>(
    createRepositoryParams(),
    moderationOperationModeProvider,
) {

    companion object {

        private fun createRepositoryParams(): AssetModerationRepositoryParams<BannerMulticardSetsRecord,
            BannerMulticardSetsStatusmoderate, BannerMulticardSetsModerationVersionsRecord> {
            return AssetModerationRepositoryParams.builder()
                .withTable(Tables.BANNER_MULTICARD_SETS)
                .withIdField(Tables.BANNER_MULTICARD_SETS.BID)
                .withStatusModerateField(Tables.BANNER_MULTICARD_SETS.STATUS_MODERATE)
                .withTransportStatusConverter(TransportStatusAdapter::toBannerMulticardSetsStatusModerate)
                .withVersionsTable(Tables.BANNER_MULTICARD_SETS_MODERATION_VERSIONS)
                .withVersionsTableIdField(Tables.BANNER_MULTICARD_SETS_MODERATION_VERSIONS.BID)
                .withVersionField(Tables.BANNER_MULTICARD_SETS_MODERATION_VERSIONS.VERSION)
                .withCreateTimeField(Tables.BANNER_MULTICARD_SETS_MODERATION_VERSIONS.CREATE_TIME)
                .build()
        }
    }

    private val bannerMulticardSetMapper = JooqMapperWithSupplierBuilder
        .builder { BannerMulticardSetWithModerationInfo() }
        .map(property(BannerMulticardSetWithModerationInfo.BID, BANNER_MULTICARDS.BID))
        .map(property(BannerMulticardSetWithModerationInfo.AD_GROUP_ID, BANNERS.PID))
        .map(property(BannerMulticardSetWithModerationInfo.CAMPAIGN_ID, BANNERS.CID))
        .map(property(BannerMulticardSetWithModerationInfo.CLIENT_ID, CAMPAIGNS.CLIENT_ID))
        .map(property(BannerMulticardSetWithModerationInfo.UID, CAMPAIGNS.UID))
        .readProperty(
            BannerMulticardSetWithModerationInfo.CLIENT_FLAGS,
            fromField(CLIENTS_OPTIONS.CLIENT_FLAGS).by(RepositoryUtils::setFromDb)
        )
        .readProperty(
            BannerMulticardSetWithModerationInfo.TRANSPORT_STATUS,
            fromField(BANNER_MULTICARD_SETS.STATUS_MODERATE).by(TransportStatusAdapter::fromDb)
        )
        .map(
            property(
                BannerMulticardSetWithModerationInfo.VERSION,
                BANNER_MULTICARD_SETS_MODERATION_VERSIONS.VERSION
            )
        )
        .map(property(BannerMulticardSetWithModerationInfo.BID_AUTO_MODERATE, AUTO_MODERATE.SEND_MULTICARD))
        .map(property(BannerMulticardSetWithModerationInfo.BID_RE_MODERATE, PRE_MODERATE_BANNERS.SEND_MULTICARD))
        .map(property(BannerMulticardSetWithModerationInfo.BANNER_TYPE, BANNERS.BANNER_TYPE))
        .readProperty(
            BannerMulticardSetWithModerationInfo.GEO,
            fromField(PHRASES.GEO).by(AdGroupMappings::geoFromDb)
        )
        .map(property(BannerMulticardSetWithModerationInfo.HREF, BANNERS.HREF))
        .map(property(BannerMulticardSetWithModerationInfo.DOMAIN, BANNERS.DOMAIN))
        .map(property(BannerMulticardSetWithModerationInfo.BODY, BANNERS.BODY))
        .map(property(BannerMulticardSetWithModerationInfo.TITLE, BANNERS.TITLE))
        .map(property(BannerMulticardSetWithModerationInfo.TITLE_EXTENSION, BANNERS.TITLE_EXTENSION))
        .map(property(BannerMulticardSetWithModerationInfo.CREATIVE_ID, BANNERS_PERFORMANCE.CREATIVE_ID))
        .map(property(BannerMulticardSetWithModerationInfo.CREATIVE_PREVIEW_URL, PERF_CREATIVES.PREVIEW_URL))
        .map(property(BannerMulticardSetWithModerationInfo.LIVE_PREVIEW_URL, PERF_CREATIVES.LIVE_PREVIEW_URL))
        .build()

    private val bannerMulticardMapper = JooqMapperWithSupplierBuilder.builder { BannerMulticardWithModerationInfo() }
        .map(property(BannerMulticardWithModerationInfo.MULTICARD_ID, BANNER_MULTICARDS.MULTICARD_ID))
        .map(property(BannerMulticardWithModerationInfo.BID, BANNER_MULTICARDS.BID))
        .map(property(BannerMulticardWithModerationInfo.ORDER, BANNER_MULTICARDS.ORDER_NUM))
        .map(property(BannerMulticardWithModerationInfo.TEXT, BANNER_MULTICARDS.TEXT))
        .map(property(BannerMulticardWithModerationInfo.HREF, BANNER_MULTICARDS.HREF))
        .map(property(BannerMulticardWithModerationInfo.IMAGE_HASH, BANNER_MULTICARDS.IMAGE_HASH))
        .readProperty(
            BannerMulticardWithModerationInfo.MDS_GROUP_ID,
            fromField(BANNER_IMAGES_FORMATS.MDS_GROUP_ID).by { it?.toInt() ?: 0 }
        )
        .map(property(BannerMulticardWithModerationInfo.NAMESPACE, BANNER_IMAGES_FORMATS.NAMESPACE))
        .map(property(BannerMulticardWithModerationInfo.AVATARS_HOST, BANNER_IMAGES_FORMATS.AVATARS_HOST))
        .build()

    /**
     * Запросить перемодерацию мультибанеров
     */
    fun requestRemoderation(shard: Int, bannerIds: Collection<Long>) {
        if (bannerIds.isEmpty()) {
            return
        }
        dslContextProvider.ppc(shard).transaction(TransactionalRunnable { config: Configuration ->
            config.dsl().update(BANNER_MULTICARD_SETS)
                .set(BANNER_MULTICARD_SETS.STATUS_MODERATE, BannerMulticardSetsStatusmoderate.Ready)
                .where(
                    BANNER_MULTICARD_SETS.BID.`in`(bannerIds)
                        .and(BANNER_MULTICARD_SETS.STATUS_MODERATE.ne(BannerMulticardSetsStatusmoderate.New)))
                .execute()
        })
    }

    override fun getReadyObjectsCount(configuration: Configuration): Long {
        return configuration.dsl()
            .selectCount()
            .from(BANNER_MULTICARD_SETS)
            .join(BANNERS).on(BANNERS.BID.eq(BANNER_MULTICARD_SETS.BID))
            .where(BANNER_MULTICARD_SETS.STATUS_MODERATE.eq(BannerMulticardSetsStatusmoderate.Ready))
            .fetchOne(count()).toLong()
    }

    override fun loadObjectForModeration(
        lockedKeys: Collection<Long>,
        configuration: Configuration
    ): List<BannerMulticardSetWithModerationInfo> {
        val records = configuration.dsl()
            .select(bannerMulticardMapper.fieldsToRead + bannerMulticardSetMapper.fieldsToRead)
            .from(BANNER_MULTICARD_SETS)
            .join(BANNERS).on(BANNERS.BID.eq(BANNER_MULTICARD_SETS.BID))
            .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
            .join(BANNER_MULTICARDS).on(BANNER_MULTICARDS.BID.eq(BANNER_MULTICARD_SETS.BID))
            .join(BANNER_IMAGES_FORMATS).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_MULTICARDS.IMAGE_HASH))
            .leftJoin(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.BID.eq(BANNERS.BID))
            .leftJoin(PERF_CREATIVES).on(
                PERF_CREATIVES.CREATIVE_ID.eq(BANNERS_PERFORMANCE.CREATIVE_ID).and(
                    PERF_CREATIVES.CREATIVE_TYPE.`in`(
                        PerfCreativesCreativeType.video_addition,
                        PerfCreativesCreativeType.bannerstorage
                    )
                )
            )
            .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
            .leftJoin(BANNER_MULTICARD_SETS_MODERATION_VERSIONS).on(
                BANNER_MULTICARD_SETS_MODERATION_VERSIONS.BID.eq(BANNER_MULTICARD_SETS.BID)
            )
            .leftJoin(PRE_MODERATE_BANNERS).on(
                PRE_MODERATE_BANNERS.BID.eq(BANNER_MULTICARD_SETS.BID)
            )
            .leftJoin(AUTO_MODERATE).on(AUTO_MODERATE.BID.eq(BANNER_MULTICARD_SETS.BID))
            .where(BANNER_MULTICARD_SETS.BID.`in`(lockedKeys))
            .orderBy(BANNER_MULTICARDS.BID, BANNER_MULTICARDS.ORDER_NUM)
            .fetch()
        return mapBannerMulticardSet(records)
    }

    private fun mapBannerMulticardSet(records: List<Record>): List<BannerMulticardSetWithModerationInfo> {
        val multicardSets: MutableMap<Long, BannerMulticardSetWithModerationInfo> = mutableMapOf()

        for (record in records) {
            val bid = record[BANNERS.BID]

            val multicardSet = multicardSets.computeIfAbsent(bid) {
                bannerMulticardSetMapper.fromDb(record)
                    .withMulticards(mutableListOf())
            }

            multicardSet.multicards.add(bannerMulticardMapper.fromDb(record))
        }

        return multicardSets.values
            .onEach { it.multicards = it.multicards.sortedBy { multicard -> multicard.order } }
            .toList()
    }
}
