package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerVcardWithModerationInfo;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.dbschema.ppc.enums.BannersPhoneflag;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerVcardsModerationVersionsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.BANNER_VCARD;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_VCARDS_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.ORG_DETAILS;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.VCARDS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class BannerVcardSendingRepository
        extends AssetModerationRepository<BannersRecord, BannersPhoneflag,
        BannerVcardsModerationVersionsRecord, BannerVcardWithModerationInfo> {

    private final JooqMapperWithSupplier<BannerVcardWithModerationInfo> mapper;

    public BannerVcardSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(createRepositoryParams(), moderationOperationModeProvider);
        this.mapper = createMapper();
    }

    private static AssetModerationRepositoryParams<BannersRecord, BannersPhoneflag,
            BannerVcardsModerationVersionsRecord>
    createRepositoryParams() {
        return AssetModerationRepositoryParams.builder()
                .withTable(BANNERS)
                .withIdField(BANNERS.BID)
                .withStatusModerateField(BANNERS.PHONEFLAG)
                .withTransportStatusConverter(TransportStatusAdapter::toBannersPhoneflag)
                .withVersionsTable(BANNER_VCARDS_MODERATION_VERSIONS)
                .withVersionsTableIdField(BANNER_VCARDS_MODERATION_VERSIONS.BID)
                .withVersionField(BANNER_VCARDS_MODERATION_VERSIONS.VERSION)
                .withCreateTimeField(BANNER_VCARDS_MODERATION_VERSIONS.CREATE_TIME)
                .build();
    }

    protected Condition getStatusModerateCondition() {
        // в ограниченном режиме не фильтруем по статусам модерации, подходят любые
        if (getModerationOperationModeProvider().getMode(BANNER_VCARD).equals(RESTRICTED)) {
            return DSL.trueCondition();
        }
        return super.getStatusModerateCondition();
    }

    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<BannerVcardWithModerationInfo> vcards) {
        // в ограниченном режиме не обновляем статусы модерации
        if (getModerationOperationModeProvider().getMode(BANNER_VCARD).equals(RESTRICTED)) {
            return;
        }
        super.updateStatusModerate(config, from, to, vcards);
    }

    @Override
    public List<BannerVcardWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                       Configuration config) {
        return DSL.using(config)
                .select(mapper.getFieldsToRead())
                .from(BANNERS)
                .join(VCARDS).on(VCARDS.VCARD_ID.eq(BANNERS.VCARD_ID))
                .leftJoin(ORG_DETAILS).on(ORG_DETAILS.ORG_DETAILS_ID.eq(VCARDS.ORG_DETAILS_ID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .leftJoin(AUTO_MODERATE).on(AUTO_MODERATE.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_VCARDS_MODERATION_VERSIONS).on(BANNER_VCARDS_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(PRE_MODERATE_BANNERS.BID.eq(BANNERS.BID))
                .where(BANNERS.BID.in(lockedKeys))
                .fetch(mapper::fromDb);
    }

    private static JooqMapperWithSupplier<BannerVcardWithModerationInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(BannerVcardWithModerationInfo::new)
                .map(property(BannerVcardWithModerationInfo.ID, VCARDS.VCARD_ID))
                .map(property(BannerVcardWithModerationInfo.NAME, VCARDS.NAME))
                .map(property(BannerVcardWithModerationInfo.BID, BANNERS.BID))
                .map(property(BannerVcardWithModerationInfo.AD_GROUP_ID, BANNERS.PID))
                .map(property(BannerVcardWithModerationInfo.CAMPAIGN_ID, BANNERS.CID))
                .map(property(BannerVcardWithModerationInfo.LAST_CHANGE, VCARDS.LAST_CHANGE))
                .map(property(BannerVcardWithModerationInfo.WORKTIME, VCARDS.WORKTIME))
                .map(property(BannerVcardWithModerationInfo.CONTACT_PERSON, VCARDS.CONTACTPERSON))
                .map(property(BannerVcardWithModerationInfo.PHONE, VCARDS.PHONE))
                .map(property(BannerVcardWithModerationInfo.EMAIL, VCARDS.CONTACT_EMAIL))
                .map(property(BannerVcardWithModerationInfo.EXTRA_MESSAGE, VCARDS.EXTRA_MESSAGE))
                .map(property(BannerVcardWithModerationInfo.OGRN, ORG_DETAILS.OGRN))
                .map(property(BannerVcardWithModerationInfo.COUNTRY, VCARDS.COUNTRY))
                .map(property(BannerVcardWithModerationInfo.CITY, VCARDS.CITY))
                .map(property(BannerVcardWithModerationInfo.STREET, VCARDS.STREET))
                .map(property(BannerVcardWithModerationInfo.HOUSE, VCARDS.HOUSE))
                .map(property(BannerVcardWithModerationInfo.BUILD, VCARDS.BUILD))
                .map(property(BannerVcardWithModerationInfo.APART, VCARDS.APART))
                .map(property(BannerVcardWithModerationInfo.CLIENT_ID, CAMPAIGNS.CLIENT_ID))
                .map(property(BannerVcardWithModerationInfo.UID, CAMPAIGNS.UID))
                .map(property(BannerVcardWithModerationInfo.VERSION, BANNER_VCARDS_MODERATION_VERSIONS.VERSION))
                .map(property(BannerVcardWithModerationInfo.BID_AUTO_MODERATE, AUTO_MODERATE.BID))
                .map(property(BannerVcardWithModerationInfo.BID_RE_MODERATE, PRE_MODERATE_BANNERS.BID))
                .map(property(BannerVcardWithModerationInfo.IM_LOGIN, VCARDS.IM_LOGIN))
                .map(property(BannerVcardWithModerationInfo.IM_CLIENT, VCARDS.IM_CLIENT))
                .map(convertibleProperty(BannerVcardWithModerationInfo.TRANSPORT_STATUS, BANNERS.PHONEFLAG,
                        TransportStatusAdapter::fromDb,
                        TransportStatusAdapter::toBannersPhoneflag))
                .build();
    }
}
