package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Record1;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.VideoWithModerationInfo;
import ru.yandex.direct.core.entity.client.repository.ClientOptionsMapping;
import ru.yandex.direct.core.entity.creative.repository.CreativeMappings;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.dbschema.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesCreativeType;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerCreativesModerationVersionsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersPerformanceRecord;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.BANNER_VIDEO_ADDITION;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_CREATIVES_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.enums.BannersPerformanceStatusmoderate.Ready;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class BannerVideoAdditionSendingRepository
        extends AssetModerationRepository<BannersPerformanceRecord, BannersPerformanceStatusmoderate,
        BannerCreativesModerationVersionsRecord, VideoWithModerationInfo> {

    private final JooqMapperWithSupplier<VideoWithModerationInfo> mapper;
    private final TableField<BannerCreativesModerationVersionsRecord, Long> creativeIdField;

    public BannerVideoAdditionSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(createRepositoryParams(), moderationOperationModeProvider);
        this.mapper = createMapper();
        this.creativeIdField = BANNER_CREATIVES_MODERATION_VERSIONS.CREATIVE_ID;
    }

    private static AssetModerationRepositoryParams<BannersPerformanceRecord, BannersPerformanceStatusmoderate,
            BannerCreativesModerationVersionsRecord> createRepositoryParams() {
        return AssetModerationRepositoryParams.builder()
                .withTable(BANNERS_PERFORMANCE)
                .withIdField(BANNERS_PERFORMANCE.BID)
                .withStatusModerateField(BANNERS_PERFORMANCE.STATUS_MODERATE)
                .withTransportStatusConverter(TransportStatusAdapter::toBannersPerformanceStatusmoderate)
                .withVersionsTable(BANNER_CREATIVES_MODERATION_VERSIONS)
                .withVersionsTableIdField(BANNER_CREATIVES_MODERATION_VERSIONS.BID)
                .withVersionField(BANNER_CREATIVES_MODERATION_VERSIONS.VERSION)
                .withCreateTimeField(BANNER_CREATIVES_MODERATION_VERSIONS.CREATE_TIME)
                .build();
    }

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        return configuration.dsl()
                .select(DSL.count().cast(Long.class))
                .from(BANNERS_PERFORMANCE)
                .join(PERF_CREATIVES).on(PERF_CREATIVES.CREATIVE_ID.eq(BANNERS_PERFORMANCE.CREATIVE_ID))
                .where(BANNERS_PERFORMANCE.STATUS_MODERATE.eq(Ready))
                .and(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition))
                .fetchOne(Record1::value1);
    }

    @Override
    public List<VideoWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                 Configuration config) {
        return DSL.using(config)
                .select(mapper.getFieldsToRead())
                .from(BANNERS_PERFORMANCE)
                .join(PERF_CREATIVES).on(PERF_CREATIVES.CREATIVE_ID.eq(BANNERS_PERFORMANCE.CREATIVE_ID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS_PERFORMANCE.CID))
                .leftJoin(BANNER_CREATIVES_MODERATION_VERSIONS).on(
                        BANNER_CREATIVES_MODERATION_VERSIONS.BID.eq(BANNERS_PERFORMANCE.BID),
                        BANNER_CREATIVES_MODERATION_VERSIONS.CREATIVE_ID.eq(BANNERS_PERFORMANCE.CREATIVE_ID)
                )
                .leftJoin(AUTO_MODERATE).on(AUTO_MODERATE.BID.eq(BANNERS_PERFORMANCE.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(PRE_MODERATE_BANNERS.BID.eq(BANNERS_PERFORMANCE.BID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .where(BANNERS_PERFORMANCE.BID.in(lockedKeys))
                .fetch(mapper::fromDb);
    }

    @Override
    protected Condition getStatusModerateCondition() {
        // в ограниченном режиме не фильтруем по статусам модерации, подходят любые
        if (getModerationOperationModeProvider().getMode(BANNER_VIDEO_ADDITION).equals(RESTRICTED)) {
            return DSL.trueCondition();
        }
        return super.getStatusModerateCondition();
    }

    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<VideoWithModerationInfo> videoAdditions) {
        // в ограниченном режиме не обновляем статусы модерации
        if (getModerationOperationModeProvider().getMode(BANNER_VIDEO_ADDITION).equals(RESTRICTED)) {
            return;
        }
        super.updateStatusModerate(config, from, to, videoAdditions);
    }

    @Override
    public void setModerationVersions(Configuration configuration, List<Pair<VideoWithModerationInfo, Long>> inserts) {
        if (inserts.isEmpty()) {
            return;
        }

        InsertHelper<BannerCreativesModerationVersionsRecord> insertHelper = new InsertHelper<>(DSL.using(configuration),
                getParams().getVersionsTable());
        inserts.forEach(insert -> {
            insertHelper.set(getParams().getVersionsTableIdField(), insert.getLeft().getBid());
            insertHelper.set(creativeIdField, insert.getLeft().getCreativeId());
            insertHelper.set(getParams().getVersionField(), insert.getRight());
            insertHelper.newRecord();
        });

        insertHelper
                .onDuplicateKeyUpdate()
                .set(getParams().getVersionField(), MySQLDSL.values(getParams().getVersionField()))
                .set(getParams().getCreateTimeField(), MySQLDSL.currentLocalDateTime())
                .execute();
    }

    private static JooqMapperWithSupplier<VideoWithModerationInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(VideoWithModerationInfo::new)
                .map(property(VideoWithModerationInfo.BID, BANNERS_PERFORMANCE.BID))
                .map(property(VideoWithModerationInfo.CREATIVE_ID, BANNERS_PERFORMANCE.CREATIVE_ID))
                .map(property(VideoWithModerationInfo.AD_GROUP_ID, BANNERS_PERFORMANCE.PID))
                .map(property(VideoWithModerationInfo.CAMPAIGN_ID, BANNERS_PERFORMANCE.CID))
                .map(property(VideoWithModerationInfo.CLIENT_ID, CAMPAIGNS.CLIENT_ID))
                .map(property(VideoWithModerationInfo.UID, CAMPAIGNS.UID))
                .map(property(VideoWithModerationInfo.VERSION,
                        BANNER_CREATIVES_MODERATION_VERSIONS.VERSION))
                .map(property(VideoWithModerationInfo.PREVIEW_URL, PERF_CREATIVES.PREVIEW_URL))
                .map(property(VideoWithModerationInfo.LIVE_PREVIEW_URL, PERF_CREATIVES.LIVE_PREVIEW_URL))
                .map(property(VideoWithModerationInfo.DURATION, PERF_CREATIVES.DURATION))
                .map(convertibleProperty(VideoWithModerationInfo.MODERATION_INFO, PERF_CREATIVES.MODERATE_INFO,
                        CreativeMappings::moderationInfoFromDb,
                        CreativeMappings::moderationInfoToDb))
                .map(property(VideoWithModerationInfo.BID_AUTO_MODERATE, AUTO_MODERATE.BID))
                .map(property(VideoWithModerationInfo.BID_RE_MODERATE, PRE_MODERATE_BANNERS.BID))
                .map(convertibleProperty(VideoWithModerationInfo.TRANSPORT_STATUS,
                        BANNERS_PERFORMANCE.STATUS_MODERATE,
                        TransportStatusAdapter::fromDb,
                        TransportStatusAdapter::toBannersPerformanceStatusmoderate))
                .map(convertibleProperty(VideoWithModerationInfo.CLIENT_FLAGS,
                        CLIENTS_OPTIONS.CLIENT_FLAGS,
                        ClientOptionsMapping::clientFlagsFromDb,
                        ClientOptionsMapping::clientFlagsToDb))
                .build();
    }
}
