package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Record1;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings;
import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithModerationInfo;
import ru.yandex.direct.core.entity.banner.type.additionalhrefs.BannerAdditionalHrefsRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.client.repository.ClientMapping;
import ru.yandex.direct.core.entity.creative.model.SourceMediaType;
import ru.yandex.direct.core.entity.creative.repository.CreativeMappings;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesCreativeType;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

public abstract class BannerWithCreativesModerationRepository extends BannersModerationRepository<BannerWithModerationInfo> {

    private final BannerAdditionalHrefsRepository bannerAdditionalHrefsRepository;

    private final Collection<TableField<?, ?>> bannerWithCreativeModerationRequestFields;
    private final OldJooqMapperWithSupplier<BannerWithModerationInfo> bannerWithCreativeMapper;

    private static final Logger logger = LoggerFactory.getLogger(BannerWithCreativesModerationRepository.class);

    public BannerWithCreativesModerationRepository(
            BannerAdditionalHrefsRepository bannerAdditionalHrefsRepository,
            ModerationOperationModeProvider moderationOperationModeProvider) {
        super(moderationOperationModeProvider);
        this.bannerWithCreativeMapper = createBannerWithCreativeMapper();
        this.bannerWithCreativeModerationRequestFields = bannerWithCreativeMapper.getFieldsToRead();
        this.bannerAdditionalHrefsRepository = bannerAdditionalHrefsRepository;
    }

    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration configuration) {
        // https://st.yandex-team.ru/DIRECT-119796
        // https://st.yandex-team.ru/DIRECT-118520
        // Почему-то иногда у баннера может не быть креатива. Это ломает транспорт. Поэтому выберем те у которых он
        // точно есть.

        List<Long> bannersWithPerformance = DSL.using(configuration)
                .selectDistinct(BANNERS_PERFORMANCE.BID)
                .from(BANNERS_PERFORMANCE)
                .where(BANNERS_PERFORMANCE.BID.in(keys))
                .forUpdate()
                .fetch(r -> r.get(BANNERS_PERFORMANCE.BID));

        if (bannersWithPerformance.size() != keys.size()) {
            var diff = Sets.difference(Set.of(keys), Set.copyOf(bannersWithPerformance));
            logger.error("Broken '{}' banners (no creative): {}", getCreativeTypes(), diff);
        }

        return super.lockKeys(bannersWithPerformance, configuration);
    }

    protected abstract List<PerfCreativesCreativeType> getCreativeTypes();

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        return configuration.dsl()
                .select(DSL.count().cast(Long.class))
                .from(BANNERS)
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .join(BANNERS_PERFORMANCE).on(BANNERS.BID.eq(BANNERS_PERFORMANCE.BID))
                .join(PERF_CREATIVES).on(PERF_CREATIVES.CREATIVE_ID.eq(BANNERS_PERFORMANCE.CREATIVE_ID))
                .where(
                        PERF_CREATIVES.CREATIVE_TYPE.in(getCreativeTypes()),
                        BANNERS.BANNER_TYPE.eq(getBannerType()),
                        BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Ready),
                        PHRASES.ADGROUP_TYPE.in(getPhrasesAdgroupTypes())
                )
                .fetchOne(Record1::value1);
    }

    @Override
    public List<BannerWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                  Configuration configuration) {
        DSLContext context = DSL.using(configuration);
        Map<Long, BannerWithModerationInfo> bannerIdToModerationInfo = context
                .select(bannerWithCreativeModerationRequestFields)
                .from(BANNERS)
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .join(USERS).on(USERS.UID.eq(CAMPAIGNS.UID))
                .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .join(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.BID.eq(BANNERS.BID))
                .join(PERF_CREATIVES).on(PERF_CREATIVES.CREATIVE_ID.eq(BANNERS_PERFORMANCE.CREATIVE_ID).and(
                        PERF_CREATIVES.CREATIVE_TYPE.in(getCreativeTypes())))
                .leftJoin(AUTO_MODERATE).on(BANNERS.BID.eq(AUTO_MODERATE.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(BANNERS.BID.eq(PRE_MODERATE_BANNERS.BID))
                .leftJoin(BANNER_MODERATION_VERSIONS).on(BANNER_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_TURBOLANDINGS).on(BANNER_TURBOLANDINGS.BID.eq(BANNERS.BID))
                .leftJoin(TURBOLANDINGS).on(TURBOLANDINGS.TL_ID.eq(BANNER_TURBOLANDINGS.TL_ID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .leftJoin(ADGROUPS_MOBILE_CONTENT).on(ADGROUPS_MOBILE_CONTENT.PID.eq(BANNERS.PID))
                .leftJoin(MOBILE_CONTENT).on(MOBILE_CONTENT.MOBILE_CONTENT_ID.eq(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .where(BANNERS.BID.in(lockedKeys))
                .fetchMap(BANNERS.BID, bannerWithCreativeMapper::fromDb);

        Map<Long, List<BannerAdditionalHref>> bannerIdToAdditionalHrefs =
                bannerAdditionalHrefsRepository.getAdditionalHrefs(context, lockedKeys);

        return EntryStream.of(bannerIdToModerationInfo)
                .mapKeyValue((bannerId, moderationInfo) -> moderationInfo
                        .withAdditionalHrefs(bannerIdToAdditionalHrefs.getOrDefault(bannerId, emptyList())))
                .toList();

    }

    private OldJooqMapperWithSupplier<BannerWithModerationInfo> createBannerWithCreativeMapper() {
        return ModerationRepositoryMapperProvider.createCommonBannerMapperBuilder()
                .map(field(BANNERS.BANNER_ID, BannerWithModerationInfo.BS_BANNER_ID))
                .map(field(BANNER_MODERATION_VERSIONS.VERSION, BannerWithModerationInfo.VERSION))
                // Логин, кажется, больше тут не нужен, не передаем в модерацию
                .map(field(USERS.LOGIN, BannerWithModerationInfo.LOGIN))
                .map(field(BANNERS.TITLE, BannerWithModerationInfo.TITLE))
                .map(field(BANNERS.TITLE_EXTENSION, BannerWithModerationInfo.TITLE_EXTENSION))
                .map(field(BANNERS.BODY, BannerWithModerationInfo.BODY))
                .map(field(BANNERS.BANNER_TYPE, BannerWithModerationInfo.BANNER_TYPE))
                .map(field(PERF_CREATIVES.CREATIVE_ID, BannerWithModerationInfo.CREATIVE_ID))
                .map(field(PERF_CREATIVES.PREVIEW_URL, BannerWithModerationInfo.CREATIVE_PREVIEW_URL))
                .map(field(PERF_CREATIVES.LIVE_PREVIEW_URL, BannerWithModerationInfo.LIVE_PREVIEW_URL))
                .map(field(PERF_CREATIVES.WIDTH, BannerWithModerationInfo.WIDTH))
                .map(field(PERF_CREATIVES.HEIGHT, BannerWithModerationInfo.HEIGHT))
                .map(field(PERF_CREATIVES.ARCHIVE_URL, BannerWithModerationInfo.ARCHIVE_URL))
                .map(field(TURBOLANDINGS.HREF, BannerWithModerationInfo.TURBOLANDING_HREF))
                .map(convertibleField(CAMPAIGNS.TYPE, BannerWithModerationInfo.CAMPAIGN_TYPE)
                        .convertFromDbBy(CampaignType::fromSource)
                )
                .map(convertibleField(PERF_CREATIVES.SOURCE_MEDIA_TYPE, BannerWithModerationInfo.SOURCE_MEDIA_TYPE)
                        .convertToDbBy(SourceMediaType::toSource)
                        .convertFromDbBy(SourceMediaType::fromSource))
                .map(convertibleField(PERF_CREATIVES.IS_GENERATED, BannerWithModerationInfo.SIMPLE_PICTURE)
                        .convertFromDbBy(isGenerated -> Objects.equals(isGenerated, 1L)))
                // Валюту тоже, похоже, можно выпилить
                .map(convertibleField(CLIENTS.WORK_CURRENCY, BannerWithModerationInfo.CURRENCY)
                        .convertToDbBy(ClientMapping::workCurrencyToDb)
                        .convertFromDbBy(ClientMapping::workCurrencyFromDb)
                        .withDatabaseDefault())
                .map(convertibleField(PERF_CREATIVES.MODERATE_INFO, BannerWithModerationInfo.MODERATE_INFO)
                        .convertFromDbBy(CreativeMappings::moderationInfoFromDb)
                        .convertToDbBy(CreativeMappings::moderationInfoToDb))
                .map(convertibleField(PHRASES.GEO, BannerWithModerationInfo.GEO)
                        .convertToDbBy(AdGroupMappings::geoToDb)
                        .convertFromDbBy(AdGroupMappings::geoFromDb))
                .map(field(PHRASES.ADGROUP_TYPE, BannerWithModerationInfo.AD_GROUP_TYPE))
                .map(convertibleField(BANNERS.STATUS_MODERATE, BannerWithModerationInfo.TRANSPORT_STATUS)
                        .convertToDbBy(TransportStatusAdapter::toBannerStatusModerate)
                        .convertFromDbBy(TransportStatusAdapter::fromDb)
                )
                .map(field(MOBILE_CONTENT.BUNDLE_ID, BannerWithModerationInfo.BUNDLE_ID))
                .map(field(MOBILE_CONTENT.STORE_CONTENT_ID, BannerWithModerationInfo.STORE_CONTENT_ID))
                .map(field(MOBILE_CONTENT.MOBILE_CONTENT_ID, BannerWithModerationInfo.MOBILE_CONTENT_ID))
                .map(field(ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF, BannerWithModerationInfo.APP_HREF))

                .build();
    }
}
