package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.InsertSetMoreStep;
import org.jooq.InsertSetStep;
import org.jooq.Record1;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.ModerationableBanner;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerModerationVersionsRecord;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

public abstract class BannersModerationRepository<T extends ModerationableBanner> implements ModerationSendingRepository<Long, T> {

    private final ModerationOperationModeProvider moderationOperationModeProvider;

    public BannersModerationRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        this.moderationOperationModeProvider = moderationOperationModeProvider;
    }

    protected abstract BannersBannerType getBannerType();

    protected abstract List<PhrasesAdgroupType> getPhrasesAdgroupTypes();

    /**
     * Лочит баннеры на время генерации запросов модерации
     *
     * @return коллекция banner_id
     */
    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration configuration) {
        return DSL.using(configuration)
                .select(BANNERS.BID)
                .from(BANNERS)
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .leftJoin(BANNER_MODERATION_VERSIONS).on(BANNER_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .where(BANNERS.BID.in(keys))
                .and(BANNERS.BANNER_TYPE.eq(getBannerType()))
                .and(PHRASES.ADGROUP_TYPE.in(getPhrasesAdgroupTypes()))
                .and(getStatusModerateCondition())
                .forUpdate()
                .fetch(r -> r.get(BANNERS.BID));
    }

    protected Condition getStatusModerateCondition() {
        // в ограниченном режиме не фильтруем по статусам модерации, подходят любые
        if (moderationOperationModeProvider.isForcedRestrictedMode()) {
            return DSL.trueCondition();
        }

        return BANNERS.STATUS_MODERATE.in(BannersStatusmoderate.Ready, BannersStatusmoderate.Sending);
    }

    protected ModerationOperationModeProvider getModerationOperationModeProvider() {
        return moderationOperationModeProvider;
    }

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        return configuration.dsl()
                .select(DSL.count().cast(Long.class))
                .from(BANNERS)
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .where(
                        BANNERS.BANNER_TYPE.eq(getBannerType()),
                        BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Ready),
                        PHRASES.ADGROUP_TYPE.in(getPhrasesAdgroupTypes())
                )
                .fetchOne(Record1::value1);
    }

    @Override
    public void updateStatusModerate(Configuration config,
                                     TransportStatus from,
                                     TransportStatus to,
                                     Collection<T> objects) {
        // в ограниченном режиме не обновляем статусы модерации
        if (moderationOperationModeProvider.isForcedRestrictedMode()) {
            return;
        }

        BannersStatusmoderate bannersStatusmoderateFrom = TransportStatusAdapter.toBannerStatusModerate(from);
        BannersStatusmoderate bannersStatusmoderateTo = TransportStatusAdapter.toBannerStatusModerate(to);

        DSL.using(config)
                .update(BANNERS)
                .set(BANNERS.STATUS_MODERATE, bannersStatusmoderateTo)
                .where(
                        BANNERS.STATUS_MODERATE.eq(bannersStatusmoderateFrom),
                        BANNERS.BID.in(objects.stream().map(Banner::getId).collect(Collectors.toList()))
                )
                .execute();
    }

    @Override
    public void setModerationVersions(Configuration configuration, List<Pair<T, Long>> inserts) {
        // В режиме неизменямых версий не пишем в базу.
        if (moderationOperationModeProvider.isImmutableVersionMode()) {
            return;
        }

        InsertSetStep<BannerModerationVersionsRecord> insertSetStep =
                DSL.using(configuration).insertInto(BANNER_MODERATION_VERSIONS);

        InsertSetMoreStep<BannerModerationVersionsRecord> insertSetMoreStep = null;

        if (inserts.isEmpty()) {
            return;
        }

        for (Pair<T, Long> info : inserts) {

            BannerModerationVersionsRecord record = new BannerModerationVersionsRecord();
            record.setBid(info.getLeft().getId());
            record.setVersion(info.getRight());

            if (insertSetMoreStep != null) {
                insertSetStep = insertSetMoreStep.newRecord();
            }

            insertSetMoreStep = insertSetStep.set(record);
        }

        insertSetMoreStep
                .onDuplicateKeyUpdate()
                .set(BANNER_MODERATION_VERSIONS.VERSION,
                        MySQLDSL.values(BANNER_MODERATION_VERSIONS.VERSION))
                .set(BANNER_MODERATION_VERSIONS.CREATE_TIME,
                        MySQLDSL.currentLocalDateTime())
                .execute();
    }
}
