package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.InsertSetMoreStep;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.RecordMapper;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.moderation.BannerstorageCreativeWithModerationInfo;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesCreativeType;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerstorageCreativesModerationVersionsRecord;

import static ru.yandex.direct.core.entity.moderation.repository.sending.TransportStatusAdapter.toPerfCreativesStatusmoderate;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class BannerstorageCreativesSendingRepository
        implements ModerationSendingRepository<Long, BannerstorageCreativeWithModerationInfo> {

    // Пока только смарт-баннеры
    public static final Map<Integer, String> SUPPORTED_TEMPLATES = Map.of(
            740, "Smart Banner retail",
            778, "Smart Banner realty",
            741, "Smart Banner hotels",
            783, "Smart Banner flights",
            779, "Smart Banner auto",
            839, "Smart Banner clothes",
            910, "Smart Banner other",
            1051, "Smart Banner pharm"
    );

    private static final String CREATIVE_VERSION_ID_ALIAS = "creative_version_id";

    private final ModerationOperationModeProvider moderationOperationModeProvider;

    @Autowired
    public BannerstorageCreativesSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        this.moderationOperationModeProvider = moderationOperationModeProvider;
    }

    @Override
    public long getReadyObjectsCount(Configuration config) {
        return config.dsl()
                .select(DSL.count().cast(Long.class))
                .from(PERF_CREATIVES)
                .where(PERF_CREATIVES.CREATIVE_TYPE.in(
                        PerfCreativesCreativeType.bannerstorage,
                        PerfCreativesCreativeType.performance))
                .and(PERF_CREATIVES.STATUS_MODERATE.eq(PerfCreativesStatusmoderate.Ready))
                .and(PERF_CREATIVES.TEMPLATE_ID.in(SUPPORTED_TEMPLATES.keySet()))
                .fetchOne(Record1::value1);
    }

    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration config) {
        return DSL.using(config)
                .select(PERF_CREATIVES.CREATIVE_ID)
                .from(PERF_CREATIVES)
                .where(PERF_CREATIVES.CREATIVE_ID.in(keys))
                .and(getStatusModerateCondition())
                .forUpdate()
                .fetch(r -> r.get(PERF_CREATIVES.CREATIVE_ID));
    }

    @SuppressWarnings("unchecked")
    protected Condition getStatusModerateCondition() {
        // в ограниченном режиме не фильтруем по статусам модерации, подходят любые
        if (moderationOperationModeProvider.isForcedRestrictedMode()) {
            return DSL.trueCondition();
        }

        return PERF_CREATIVES.STATUS_MODERATE.in(
                PerfCreativesStatusmoderate.Ready,
                PerfCreativesStatusmoderate.Sending,
                // Для предотвращения гонки между bannerstorage и Директом,
                // пока не будет сделан более качественный фикс
                PerfCreativesStatusmoderate.Sent);
    }

    @Override
    public List<BannerstorageCreativeWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                                 Configuration config) {
        return DSL.using(config)
                .select(
                        PERF_CREATIVES.CREATIVE_ID,
                        // alias, т.к. есть ещё одно поле с названием version
                        PERF_CREATIVES.VERSION.as(CREATIVE_VERSION_ID_ALIAS),
                        PERF_CREATIVES.STATUS_MODERATE,
                        // Берём последнюю версию или null, если версий ещё нет
                        DSL.max(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS.VERSION)
                                .as(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS.VERSION),
                        // Остальные поля берём одни из, т.к. нет возможности понять, в привязке к какой из кампаний
                        // был изменён креатив (впрочем, это и не важно)
                        DSL.min(CAMPAIGNS.CLIENT_ID).as(CAMPAIGNS.CLIENT_ID),
                        DSL.min(CAMPAIGNS.UID).as(CAMPAIGNS.UID),
                        DSL.min(BANNERS.CID).as(BANNERS.CID),
                        DSL.min(BANNERS.PID).as(BANNERS.PID),
                        DSL.min(BANNERS.BID).as(BANNERS.BID)
                )
                .from(PERF_CREATIVES)
                .join(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .join(BANNERS).on(BANNERS.BID.eq(BANNERS_PERFORMANCE.BID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .leftJoin(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS).on(
                        BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .where(PERF_CREATIVES.CREATIVE_ID.in(lockedKeys))
                .groupBy(PERF_CREATIVES.CREATIVE_ID, PERF_CREATIVES.VERSION, PERF_CREATIVES.STATUS_MODERATE)
                .fetch(new BannerstorageCreativeRecordMapper());
    }

    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<BannerstorageCreativeWithModerationInfo> loadedObjects) {
        // в ограниченном режиме не обновляем статусы модерации
        if (moderationOperationModeProvider.isForcedRestrictedMode()) {
            return;
        }

        DSL.using(config)
                .update(PERF_CREATIVES)
                .set(PERF_CREATIVES.STATUS_MODERATE, toPerfCreativesStatusmoderate(to))
                .where(
                        PERF_CREATIVES.STATUS_MODERATE.eq(toPerfCreativesStatusmoderate(from)),
                        PERF_CREATIVES.CREATIVE_ID
                                .in(mapList(loadedObjects, BannerstorageCreativeWithModerationInfo::getCreativeId)))
                .execute();
    }

    @Override
    public void setModerationVersions(Configuration configuration,
                                      List<Pair<BannerstorageCreativeWithModerationInfo, Long>> inserts) {
        if (inserts.isEmpty()) {
            return;
        }

        var insertSetStep = DSL.using(configuration).insertInto(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS);
        InsertSetMoreStep<BannerstorageCreativesModerationVersionsRecord> insertSetMoreStep = null;
        for (Pair<BannerstorageCreativeWithModerationInfo, Long> info : inserts) {
            var record = new BannerstorageCreativesModerationVersionsRecord();
            record.setBid(info.getLeft().getBid());
            record.setVersion(info.getRight());

            if (insertSetMoreStep != null) {
                insertSetStep = insertSetMoreStep.newRecord();
            }

            insertSetMoreStep = insertSetStep.set(record);
        }

        insertSetMoreStep
                .onDuplicateKeyUpdate()
                .set(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS.VERSION,
                        MySQLDSL.values(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS.VERSION))
                .set(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS.CREATE_TIME,
                        MySQLDSL.currentLocalDateTime())
                .execute();
    }

    private static class BannerstorageCreativeRecordMapper implements RecordMapper<Record,
            BannerstorageCreativeWithModerationInfo> {
        @Override
        public BannerstorageCreativeWithModerationInfo map(Record record) {
            return new BannerstorageCreativeWithModerationInfo()
                    .withClientId(record.get(CAMPAIGNS.CLIENT_ID))
                    .withUid(record.get(CAMPAIGNS.UID))
                    .withCampaignId(record.get(BANNERS.CID))
                    .withAdGroupId(record.get(BANNERS.PID))
                    .withBid(record.get(BANNERS.BID))
                    .withCreativeId(record.get(PERF_CREATIVES.CREATIVE_ID))
                    .withCreativeVersionId((Long) record.get(CREATIVE_VERSION_ID_ALIAS))
                    .withTransportStatus(TransportStatusAdapter.fromDb(record.get(PERF_CREATIVES.STATUS_MODERATE)))
                    .withVersion(record.get(BANNERSTORAGE_CREATIVES_MODERATION_VERSIONS.VERSION))
                    .withBidAutoModerate(null)
                    .withBidReModerate(null);
        }
    }
}
