package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Record1;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.CalloutModerateType;
import ru.yandex.direct.core.entity.banner.model.CalloutWithModerationInfo;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.dbschema.ppc.enums.AdditionsItemCalloutsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.CalloutModerationVersionsRecord;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.CALLOUT;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDITIONS_ITEM_CALLOUTS;
import static ru.yandex.direct.dbschema.ppc.Tables.CALLOUT_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.MODERATE_ADDITIONS;
import static ru.yandex.direct.dbschema.ppc.enums.AdditionsItemCalloutsStatusmoderate.Ready;
import static ru.yandex.direct.dbschema.ppc.enums.AdditionsItemCalloutsStatusmoderate.Sending;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class CalloutSendingRepository implements ModerationSendingRepository<Long, CalloutWithModerationInfo> {

    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final JooqMapperWithSupplier<CalloutWithModerationInfo> mapper;

    public CalloutSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        mapper = createMapper();
    }

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        return configuration.dsl()
                .select(DSL.count().cast(Long.class))
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE.eq(Ready))
                .fetchOne(Record1::value1);
    }

    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration config) {
        return DSL.using(config)
                .select(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(keys))
                .and(getStatusModerateCondition())
                .forUpdate()
                .fetch(r -> r.get(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID));
    }

    @Override
    public List<CalloutWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                   Configuration config) {
        return DSL.using(config)
                .select(mapper.getFieldsToRead())
                .from(ADDITIONS_ITEM_CALLOUTS)
                .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(ADDITIONS_ITEM_CALLOUTS.CLIENT_ID))
                .leftJoin(MODERATE_ADDITIONS)
                .on(MODERATE_ADDITIONS.ADDITIONS_ITEM_ID.eq(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID))
                .leftJoin(CALLOUT_MODERATION_VERSIONS)
                .on(CALLOUT_MODERATION_VERSIONS.CALLOUT_ID.eq(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID))
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(lockedKeys))
                .fetch(mapper::fromDb);
    }

    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<CalloutWithModerationInfo> callouts) {
        // в ограниченном режиме не обновляем статусы модерации
        if (moderationOperationModeProvider.getMode(CALLOUT).equals(RESTRICTED)) {
            return;
        }

        AdditionsItemCalloutsStatusmoderate fromStatusmoderate =
                TransportStatusAdapter.toAdditionsItemCalloutsStatusmoderate(from);

        AdditionsItemCalloutsStatusmoderate toStatusmoderate =
                TransportStatusAdapter.toAdditionsItemCalloutsStatusmoderate(to);

        DSL.using(config)
                .update(ADDITIONS_ITEM_CALLOUTS)
                .set(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE, toStatusmoderate)
                .where(
                        ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE.eq(fromStatusmoderate),
                        ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(
                                mapList(callouts, CalloutWithModerationInfo::getCalloutId)))
                .execute();
    }

    @Override
    public void setModerationVersions(Configuration configuration,
                                      List<Pair<CalloutWithModerationInfo, Long>> inserts) {
        if (inserts.isEmpty()) {
            return;
        }

        InsertHelper<CalloutModerationVersionsRecord> insertHelper = new InsertHelper<>(DSL.using(configuration),
                CALLOUT_MODERATION_VERSIONS);
        inserts.forEach(insert -> {
            insertHelper.set(CALLOUT_MODERATION_VERSIONS.CALLOUT_ID, insert.getLeft().getCalloutId());
            insertHelper.set(CALLOUT_MODERATION_VERSIONS.VERSION, insert.getRight());
            insertHelper.newRecord();
        });

        insertHelper
                .onDuplicateKeyUpdate()
                .set(CALLOUT_MODERATION_VERSIONS.VERSION, MySQLDSL.values(CALLOUT_MODERATION_VERSIONS.VERSION))
                .set(CALLOUT_MODERATION_VERSIONS.CREATE_TIME, MySQLDSL.currentLocalDateTime())
                .execute();
    }

    protected Condition getStatusModerateCondition() {
        // в ограниченном режиме не фильтруем по статусам модерации, подходят любые
        if (moderationOperationModeProvider.getMode(CALLOUT).equals(RESTRICTED)) {
            return DSL.trueCondition();
        }
        return ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE.in(Ready, Sending);
    }

    private static JooqMapperWithSupplier<CalloutWithModerationInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(CalloutWithModerationInfo::new)
                .map(property(CalloutWithModerationInfo.CALLOUT_ID, ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID))
                .map(property(CalloutWithModerationInfo.CLIENT_ID, ADDITIONS_ITEM_CALLOUTS.CLIENT_ID))
                .map(property(CalloutWithModerationInfo.TEXT, ADDITIONS_ITEM_CALLOUTS.CALLOUT_TEXT))
                .map(property(CalloutWithModerationInfo.UID, CLIENTS.CHIEF_UID))
                .map(convertibleProperty(CalloutWithModerationInfo.TRANSPORT_STATUS,
                        ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE,
                        TransportStatusAdapter::fromDb,
                        TransportStatusAdapter::toAdditionsItemCalloutsStatusmoderate))
                .map(convertibleProperty(CalloutWithModerationInfo.MODERATE_TYPE,
                        MODERATE_ADDITIONS.MODERATE_TYPE,
                        CalloutModerateType::fromSource,
                        CalloutModerateType::toSource))
                .map(property(CalloutWithModerationInfo.VERSION, CALLOUT_MODERATION_VERSIONS.VERSION))
                .build();
    }
}
