package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Record1;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithModerationInfo;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.client.repository.ClientOptionsMapping;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignModerationVersionsRecord;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.CAMPAIGN;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGN_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusmoderate.Ready;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusmoderate.Sent;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Repository
public class CampaignSendingRepository implements ModerationSendingRepository<Long, CampaignWithModerationInfo> {
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final PpcProperty<Set<Long>> bsAgencyIdsForDirectMonitoringProperty;
    private final JooqMapperWithSupplier<CampaignWithModerationInfo> mapper;

    public CampaignSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider,
                                     PpcPropertiesSupport ppcPropertiesSupport) {
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;

        this.bsAgencyIdsForDirectMonitoringProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.BS_AGENCY_IDS_FOR_DIRECT_MONITORING, Duration.ofMinutes(1));

        this.mapper = createMapper();
    }

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        return configuration.dsl()
                .select(DSL.count().cast(Long.class))
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.STATUS_MODERATE.eq(Ready))
                .and(CAMPAIGNS.TYPE.in(mapSet(CampaignTypeKinds.MOD_EXPORT, CampaignType::toSource)))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .fetchOne(Record1::value1);
    }

    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration config) {
        var query = DSL.using(config)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(keys))
                .and(getStatusModerateCondition())
                .and(CAMPAIGNS.TYPE.in(mapSet(CampaignTypeKinds.MOD_EXPORT, CampaignType::toSource)))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No));

        var bsAgencyIdsForDirectMonitoring = bsAgencyIdsForDirectMonitoringProperty.getOrDefault(Collections.emptySet());
        if (!bsAgencyIdsForDirectMonitoring.isEmpty()) {
            query = query.andNot(CAMPAIGNS.AGENCY_ID.in(bsAgencyIdsForDirectMonitoring));
        }

        return query
                .forUpdate()
                .fetch(r -> r.get(CAMPAIGNS.CID));
    }

    @Override
    public List<CampaignWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                    Configuration config) {
        return DSL.using(config)
                .select(mapper.getFieldsToRead())
                .from(CAMPAIGNS)
                .join(USERS).on(USERS.UID.eq(CAMPAIGNS.UID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .leftJoin(CAMPAIGN_MODERATION_VERSIONS).on(CAMPAIGN_MODERATION_VERSIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CID.in(lockedKeys))
                .fetch(mapper::fromDb);
    }

    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<CampaignWithModerationInfo> campaigns) {
        // в ограниченном режиме не обновляем статусы модерации
        if (moderationOperationModeProvider.getMode(CAMPAIGN).equals(RESTRICTED)) {
            return;
        }

        CampaignsStatusmoderate fromStatusmoderate = TransportStatusAdapter.toCampaignStatusmoderate(from);
        CampaignsStatusmoderate toStatusmoderate = TransportStatusAdapter.toCampaignStatusmoderate(to);

        DSL.using(config)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_MODERATE, toStatusmoderate)
                .where(
                        CAMPAIGNS.STATUS_MODERATE.eq(fromStatusmoderate),
                        CAMPAIGNS.CID.in(mapList(campaigns, CampaignWithModerationInfo::getId)))
                .execute();
    }

    @Override
    public void setModerationVersions(Configuration configuration,
                                      List<Pair<CampaignWithModerationInfo, Long>> inserts) {
        if (inserts.isEmpty()) {
            return;
        }

        InsertHelper<CampaignModerationVersionsRecord> insertHelper = new InsertHelper<>(DSL.using(configuration),
                CAMPAIGN_MODERATION_VERSIONS);
        inserts.forEach(insert -> {
            insertHelper.set(CAMPAIGN_MODERATION_VERSIONS.CID, insert.getLeft().getId());
            insertHelper.set(CAMPAIGN_MODERATION_VERSIONS.VERSION, insert.getRight());
            insertHelper.newRecord();
        });

        insertHelper
                .onDuplicateKeyUpdate()
                .set(CAMPAIGN_MODERATION_VERSIONS.VERSION, MySQLDSL.values(CAMPAIGN_MODERATION_VERSIONS.VERSION))
                .set(CAMPAIGN_MODERATION_VERSIONS.CREATE_TIME, MySQLDSL.currentLocalDateTime())
                .execute();
    }

    private Condition getStatusModerateCondition() {
        // В ограниченном режиме не фильтруем по статусам модерации, подходят любые
        if (moderationOperationModeProvider.getMode(CAMPAIGN).equals(RESTRICTED)) {
            return DSL.trueCondition();
        }
        // Ess-транспорт баннеров и ассетов агрессивно выставляют для кампании статус Sent при собственной отправке.
        // Чтобы не просыпать отправку таких кампаний, будем доставать кампании в статусе Sent тоже
        // (полагаясь на то, что событиев бинлоге пришло в статусе Ready).
        return CAMPAIGNS.STATUS_MODERATE.in(Ready, Sent);
    }

    private static JooqMapperWithSupplier<CampaignWithModerationInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(CampaignWithModerationInfo::new)
                .map(property(CampaignWithModerationInfo.ID, CAMPAIGNS.CID))
                .map(property(CampaignWithModerationInfo.NAME, CAMPAIGNS.NAME))
                .map(convertibleProperty(CampaignWithModerationInfo.TYPE,
                        CAMPAIGNS.TYPE,
                        CampaignType::fromSource,
                        CampaignType::toSource))
                .map(convertibleProperty(CampaignWithModerationInfo.STRATEGY_DATA,
                        CAMPAIGNS.STRATEGY_DATA,
                        CampaignMappings::strategyDataFromDb,
                        CampaignMappings::strategyDataToDb))
                .map(property(CampaignWithModerationInfo.CLIENT_ID, CAMPAIGNS.CLIENT_ID))
                .map(property(CampaignWithModerationInfo.UID, CAMPAIGNS.UID))
                .map(convertibleProperty(CampaignWithModerationInfo.CLIENT_FLAGS,
                        CLIENTS_OPTIONS.CLIENT_FLAGS,
                        ClientOptionsMapping::clientFlagsFromDb,
                        ClientOptionsMapping::clientFlagsToDb))
                .map(convertibleProperty(CampaignWithModerationInfo.TRANSPORT_STATUS,
                        CAMPAIGNS.STATUS_MODERATE,
                        TransportStatusAdapter::fromDb,
                        TransportStatusAdapter::toCampaignStatusmoderate))
                .map(property(CampaignWithModerationInfo.VERSION, CAMPAIGN_MODERATION_VERSIONS.VERSION))
                .map(property(CampaignWithModerationInfo.EMAIL, USERS.EMAIL))
                .map(property(CampaignWithModerationInfo.FIO, USERS.FIO))
                .map(property(CampaignWithModerationInfo.LOGIN, USERS.LOGIN))
                .build();
    }
}
