package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Configuration;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.core.entity.banner.model.BannerWithModerationInfo;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;

import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_CONTENT_PROMOTION;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CONTENT_PROMOTION;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.enums.BannersBannerType.content_promotion;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

@Repository
public class ContentPromotionVerdictRepository extends BannersModerationRepository<BannerWithModerationInfo> {
    private final Collection<TableField<?, ?>> moderationBaseRequestFields;
    private final OldJooqMapperWithSupplier<BannerWithModerationInfo> baseBannerMapper;

    @Autowired
    public ContentPromotionVerdictRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(moderationOperationModeProvider);
        this.baseBannerMapper = createBaseBannerMapper();
        this.moderationBaseRequestFields = baseBannerMapper.getFieldsToRead();
    }

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        return 0L;
    }

    @Override
    protected BannersBannerType getBannerType() {
        return content_promotion;
    }

    @Override
    protected List<PhrasesAdgroupType> getPhrasesAdgroupTypes() {
        return List.of(PhrasesAdgroupType.content_promotion);
    }

    // Переопределяем методы, потому что нам не нужно инкрементить номер версии баннера и не нужно менять
    // статус модерации на Sent, в отличие от обычной отправки на модерацию
    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<BannerWithModerationInfo> objects) {
    }

    @Override
    public void setModerationVersions(Configuration configuration, List<Pair<BannerWithModerationInfo, Long>> inserts) {
    }

    /**
     * Локает баннеры для запросов вердиктов в модерацию.
     *
     * @param keys           id баннеров
     * @param configuration конфигурация
     * @return список id баннеров
     */
    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration configuration) {
        return DSL.using(configuration)
                .select(BANNERS.BID, BANNER_MODERATION_VERSIONS.VERSION)
                .from(BANNERS)
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .leftJoin(BANNER_MODERATION_VERSIONS).on(BANNER_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .where(BANNERS.BID.in(keys))
                .and(BANNERS.BANNER_TYPE.eq(getBannerType()))
                .and(PHRASES.ADGROUP_TYPE.in(getPhrasesAdgroupTypes()))
                .forUpdate()
                .fetch(r -> r.get(BANNERS.BID));
    }

    /**
     * Возвращает метаданные баннеров для модерации (cid, pid, версия баннера и флаг автомодерации).
     *
     * @param configuration шард
     * @param lockedKeys     id баннеров
     * @return метаданные баннеров
     */
    @Override
    public List<BannerWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                  Configuration configuration) {
        return DSL.using(configuration)
                .select(moderationBaseRequestFields)
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .join(BANNERS_CONTENT_PROMOTION).on(BANNERS.BID.eq(BANNERS_CONTENT_PROMOTION.BID))
                .join(CONTENT_PROMOTION).on(CONTENT_PROMOTION.ID.eq(BANNERS_CONTENT_PROMOTION.CONTENT_PROMOTION_ID))
                .leftJoin(AUTO_MODERATE).on(BANNERS.BID.eq(AUTO_MODERATE.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(BANNERS.BID.eq(PRE_MODERATE_BANNERS.BID))
                .leftJoin(BANNER_MODERATION_VERSIONS).on(BANNER_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .where(BANNERS.BID.in(lockedKeys))
                .fetch(rec -> baseBannerMapper.fromDb(new BannerWithModerationInfo(), rec));
    }

    private OldJooqMapperWithSupplier<BannerWithModerationInfo> createBaseBannerMapper() {
        return ModerationRepositoryMapperProvider.createCommonVerdictRequestMapperBuilder()
                .map(field(BANNER_MODERATION_VERSIONS.VERSION, BannerWithModerationInfo.VERSION))
                .map(field(BANNERS.BANNER_ID, BannerWithModerationInfo.BS_BANNER_ID))
                .map(convertibleField(CONTENT_PROMOTION.TYPE, BannerWithModerationInfo.CONTENT_PROMOTION_TYPE)
                        .convertToDbBy(ContentPromotionContentType::toSource)
                        .convertFromDbBy(ContentPromotionContentType::fromSource))
                .map(convertibleField(BANNERS.STATUS_MODERATE, BannerWithModerationInfo.TRANSPORT_STATUS)
                        .convertToDbBy(TransportStatusAdapter::toBannerStatusModerate)
                        .convertFromDbBy(TransportStatusAdapter::fromDb)
                )
                .build();
    }
}
