package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import one.util.streamex.EntryStream;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings;
import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithModerationInfo;
import ru.yandex.direct.core.entity.banner.type.additionalhrefs.BannerAdditionalHrefsRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.client.repository.ClientMapping;
import ru.yandex.direct.core.entity.creative.model.SourceMediaType;
import ru.yandex.direct.core.entity.creative.repository.CreativeMappings;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

@Repository
public class CpmFrontpageBannersRepository extends BannersModerationRepository<BannerWithModerationInfo> {

    private final BannerAdditionalHrefsRepository bannerAdditionalHrefsRepository;
    private final Collection<TableField<?, ?>> moderationYndxFrontpageRequestFields;
    private final OldJooqMapperWithSupplier<BannerWithModerationInfo> cpmYndxFrontpageBannerMapper;

    @Autowired
    public CpmFrontpageBannersRepository(BannerAdditionalHrefsRepository bannerAdditionalHrefsRepository,
                                         ModerationOperationModeProvider moderationOperationModeProvider) {
        super(moderationOperationModeProvider);
        this.bannerAdditionalHrefsRepository = bannerAdditionalHrefsRepository;
        this.cpmYndxFrontpageBannerMapper = createCpmYndxFrontpageBannerMapper();
        this.moderationYndxFrontpageRequestFields = cpmYndxFrontpageBannerMapper.getFieldsToRead();
    }

    @Override
    protected BannersBannerType getBannerType() {
        return BannersBannerType.cpm_banner;
    }

    @Override
    protected List<PhrasesAdgroupType> getPhrasesAdgroupTypes() {
        return List.of(PhrasesAdgroupType.cpm_yndx_frontpage);
    }

    @Override
    public List<BannerWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                  Configuration configuration) {
        DSLContext context = DSL.using(configuration);

        Map<Long, BannerWithModerationInfo> bannerIdToModerationInfo = context
                .select(moderationYndxFrontpageRequestFields)
                .select(BANNERS.BID)
                .from(BANNERS)
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .join(USERS).on(USERS.UID.eq(CAMPAIGNS.UID))
                .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .join(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.BID.eq(BANNERS.BID))
                .join(PERF_CREATIVES).on(PERF_CREATIVES.CREATIVE_ID.eq(BANNERS_PERFORMANCE.CREATIVE_ID))
                .leftJoin(AUTO_MODERATE).on(BANNERS.BID.eq(AUTO_MODERATE.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(BANNERS.BID.eq(PRE_MODERATE_BANNERS.BID))
                .leftJoin(BANNER_MODERATION_VERSIONS).on(BANNER_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_TURBOLANDINGS).on(BANNER_TURBOLANDINGS.BID.eq(BANNERS.BID))
                .leftJoin(TURBOLANDINGS).on(TURBOLANDINGS.TL_ID.eq(BANNER_TURBOLANDINGS.TL_ID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .where(BANNERS.BID.in(lockedKeys))
                .fetchMap(BANNERS.BID, cpmYndxFrontpageBannerMapper::fromDb);

        Map<Long, List<BannerAdditionalHref>> bannerIdToAdditionalHrefs =
                bannerAdditionalHrefsRepository.getAdditionalHrefs(context, lockedKeys);

        return EntryStream.of(bannerIdToModerationInfo)
                .mapKeyValue((bannerId, moderationInfo) -> moderationInfo
                        .withAdditionalHrefs(bannerIdToAdditionalHrefs.getOrDefault(bannerId, emptyList())))
                .toList();
    }

    private OldJooqMapperWithSupplier<BannerWithModerationInfo> createCpmYndxFrontpageBannerMapper() {
        return ModerationRepositoryMapperProvider.createCommonBannerMapperBuilder()
                .map(field(BANNER_MODERATION_VERSIONS.VERSION, BannerWithModerationInfo.VERSION))
                .map(field(USERS.LOGIN, BannerWithModerationInfo.LOGIN))
                .map(field(BANNERS.BANNER_ID, BannerWithModerationInfo.BS_BANNER_ID))
                .map(convertibleField(CAMPAIGNS.TYPE, BannerWithModerationInfo.CAMPAIGN_TYPE)
                        .convertToDbBy(CampaignType::toSource)
                        .convertFromDbBy(CampaignType::fromSource))

                .map(field(PERF_CREATIVES.CREATIVE_ID, BannerWithModerationInfo.CREATIVE_ID))
                .map(field(PERF_CREATIVES.PREVIEW_URL, BannerWithModerationInfo.CREATIVE_PREVIEW_URL))
                .map(field(PERF_CREATIVES.LIVE_PREVIEW_URL, BannerWithModerationInfo.LIVE_PREVIEW_URL))
                .map(field(PERF_CREATIVES.EXPANDED_PREVIEW_URL, BannerWithModerationInfo.EXPANDED_PREVIEW_URL))
                .map(field(PERF_CREATIVES.WIDTH, BannerWithModerationInfo.WIDTH))
                .map(field(PERF_CREATIVES.HEIGHT, BannerWithModerationInfo.HEIGHT))
                .map(field(PERF_CREATIVES.ARCHIVE_URL, BannerWithModerationInfo.ARCHIVE_URL))
                .map(convertibleField(PERF_CREATIVES.SOURCE_MEDIA_TYPE, BannerWithModerationInfo.SOURCE_MEDIA_TYPE)
                        .convertToDbBy(SourceMediaType::toSource)
                        .convertFromDbBy(SourceMediaType::fromSource))
                .map(convertibleField(PERF_CREATIVES.IS_GENERATED, BannerWithModerationInfo.SIMPLE_PICTURE)
                        .convertFromDbBy(isGenerated -> Objects.equals(isGenerated, 1L)))
                .map(convertibleField(PERF_CREATIVES.MODERATE_INFO, BannerWithModerationInfo.MODERATE_INFO)
                        .convertToDbBy(CreativeMappings::moderationInfoToDb)
                        .convertFromDbBy(CreativeMappings::moderationInfoFromDb))
                .map(field(PERF_CREATIVES.DURATION, BannerWithModerationInfo.DURATION))
                .map(convertibleField(CLIENTS.WORK_CURRENCY, BannerWithModerationInfo.CURRENCY)
                        .convertToDbBy(ClientMapping::workCurrencyToDb)
                        .convertFromDbBy(ClientMapping::workCurrencyFromDb)
                        .withDatabaseDefault())
                .map(convertibleField(PHRASES.GEO, BannerWithModerationInfo.GEO)
                        .convertToDbBy(AdGroupMappings::geoToDb)
                        .convertFromDbBy(AdGroupMappings::geoFromDb))
                .map(field(TURBOLANDINGS.HREF, BannerWithModerationInfo.TURBOLANDING_HREF))
                .map(convertibleField(BANNERS.STATUS_MODERATE, BannerWithModerationInfo.TRANSPORT_STATUS)
                        .convertToDbBy(TransportStatusAdapter::toBannerStatusModerate)
                        .convertFromDbBy(TransportStatusAdapter::fromDb)
                )

                .build();
    }
}
