package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.RecordMapper;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.displayhrefs.model.DisplayHrefWithModerationInfo;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.dbschema.ppc.enums.BannerDisplayHrefsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerDisplayHrefsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerDisplayhrefsModerationVersionsRecord;

import static java.util.function.Function.identity;
import static ru.yandex.direct.common.util.RepositoryUtils.setFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerDisplayhrefsModerationVersions.BANNER_DISPLAYHREFS_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

@Repository
public class DisplayHrefsSendingRepository
        extends AssetModerationRepository<BannerDisplayHrefsRecord, BannerDisplayHrefsStatusmoderate,
        BannerDisplayhrefsModerationVersionsRecord, DisplayHrefWithModerationInfo> {

    public DisplayHrefsSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(createRepositoryParams(), moderationOperationModeProvider);
    }

    private static AssetModerationRepositoryParams<BannerDisplayHrefsRecord, BannerDisplayHrefsStatusmoderate,
            BannerDisplayhrefsModerationVersionsRecord> createRepositoryParams() {
        return AssetModerationRepositoryParams.builder()
                .withTable(BANNER_DISPLAY_HREFS)
                .withIdField(BANNER_DISPLAY_HREFS.BID)
                .withStatusModerateField(BANNER_DISPLAY_HREFS.STATUS_MODERATE)
                .withTransportStatusConverter(TransportStatusAdapter::toBannerDisplayHrefsStatusmoderate)
                .withVersionsTable(BANNER_DISPLAYHREFS_MODERATION_VERSIONS)
                .withVersionsTableIdField(BANNER_DISPLAYHREFS_MODERATION_VERSIONS.BID)
                .withVersionField(BANNER_DISPLAYHREFS_MODERATION_VERSIONS.VERSION)
                .withCreateTimeField(BANNER_DISPLAYHREFS_MODERATION_VERSIONS.CREATE_TIME)
                .build();
    }

    @Override
    public List<DisplayHrefWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                       Configuration config) {
        return DSL.using(config)
                .select(BANNER_DISPLAY_HREFS.BID,
                        BANNERS.PID,
                        BANNERS.CID,
                        CAMPAIGNS.CLIENT_ID,
                        CAMPAIGNS.UID,
                        BANNER_DISPLAY_HREFS.DISPLAY_HREF,
                        BANNER_DISPLAYHREFS_MODERATION_VERSIONS.VERSION,
                        PRE_MODERATE_BANNERS.BID,
                        AUTO_MODERATE.BID,
                        BANNER_DISPLAY_HREFS.STATUS_MODERATE,
                        CLIENTS_OPTIONS.CLIENT_FLAGS)
                .from(BANNER_DISPLAY_HREFS)
                .join(BANNERS).on(BANNER_DISPLAY_HREFS.BID.eq(BANNERS.BID))
                .join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(CLIENTS_OPTIONS).on(CAMPAIGNS.CLIENT_ID.eq(CLIENTS_OPTIONS.CLIENT_ID))
                .leftJoin(BANNER_DISPLAYHREFS_MODERATION_VERSIONS).on(BANNER_DISPLAYHREFS_MODERATION_VERSIONS.BID.eq(BANNER_DISPLAY_HREFS.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(BANNER_DISPLAY_HREFS.BID.eq(PRE_MODERATE_BANNERS.BID))
                .leftJoin(AUTO_MODERATE).on(BANNER_DISPLAY_HREFS.BID.eq(AUTO_MODERATE.BID))
                .where(BANNER_DISPLAY_HREFS.BID.in(lockedKeys))
                .fetch(new DisplayHrefsRecordMapper());
    }

    private static class DisplayHrefsRecordMapper implements RecordMapper<Record, DisplayHrefWithModerationInfo> {
        @Override
        public DisplayHrefWithModerationInfo map(Record record) {
            return new DisplayHrefWithModerationInfo()
                    .withId(record.get(BANNER_DISPLAY_HREFS.BID))
                    .withAdGroupId(record.get(BANNERS.PID))
                    .withBid(record.get(BANNER_DISPLAY_HREFS.BID))
                    .withCampaignId(record.get(BANNERS.CID))
                    .withClientId(record.get(CAMPAIGNS.CLIENT_ID))
                    .withUid(record.get(CAMPAIGNS.UID))
                    .withText(record.get(BANNER_DISPLAY_HREFS.DISPLAY_HREF))
                    .withBidAutoModerate(record.get(AUTO_MODERATE.BID))
                    .withBidReModerate(record.get(PRE_MODERATE_BANNERS.BID))
                    .withTransportStatus(TransportStatusAdapter.fromDb(record.get(BANNER_DISPLAY_HREFS.STATUS_MODERATE)))
                    .withVersion(record.get(BANNER_DISPLAYHREFS_MODERATION_VERSIONS.VERSION))
                    .withClientFlags(setFromDb(record.get(CLIENTS_OPTIONS.CLIENT_FLAGS), identity()));
        }
    }

}
