package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.RecordMapper;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.image.model.ImageWithModerationInfo;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerImagesModerationVersionsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerImagesRecord;

import static java.util.function.Function.identity;
import static ru.yandex.direct.common.util.RepositoryUtils.setFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES_FORMATS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

@Repository
public class ImageSendingRepository
        extends AssetModerationRepository<BannerImagesRecord, BannerImagesStatusmoderate,
        BannerImagesModerationVersionsRecord, ImageWithModerationInfo> {

    public ImageSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(createRepositoryParams(), moderationOperationModeProvider);
    }

    private static AssetModerationRepositoryParams<BannerImagesRecord, BannerImagesStatusmoderate,
            BannerImagesModerationVersionsRecord> createRepositoryParams() {
        return AssetModerationRepositoryParams.builder()
                .withTable(BANNER_IMAGES)
                .withIdField(BANNER_IMAGES.BID)
                .withStatusModerateField(BANNER_IMAGES.STATUS_MODERATE)
                .withTransportStatusConverter(TransportStatusAdapter::toBannerImagesStatusmoderate)
                .withVersionsTable(BANNER_IMAGES_MODERATION_VERSIONS)
                .withVersionsTableIdField(BANNER_IMAGES_MODERATION_VERSIONS.BID)
                .withVersionField(BANNER_IMAGES_MODERATION_VERSIONS.VERSION)
                .withCreateTimeField(BANNER_IMAGES_MODERATION_VERSIONS.CREATE_TIME)
                .build();
    }

    // оверрайдим, потому что в качестве ids приходят image_id, а не bid
    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration config) {
        return DSL.using(config)
                .select(BANNER_IMAGES.IMAGE_ID)
                .from(BANNER_IMAGES)
                .where(BANNER_IMAGES.IMAGE_ID.in(keys))
                .and(getStatusModerateCondition())
                .forUpdate()
                .fetch(r -> r.get(BANNER_IMAGES.IMAGE_ID));
    }

    @Override
    public List<ImageWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                 Configuration config) {
        return DSL.using(config)
                .select(BANNER_IMAGES.BID,
                        BANNER_IMAGES.IMAGE_ID,
                        BANNERS.PID,
                        BANNERS.CID,
                        CAMPAIGNS.CLIENT_ID,
                        CAMPAIGNS.UID,
                        CAMPAIGNS.TYPE,
                        BANNER_IMAGES.IMAGE_HASH,
                        BANNER_IMAGES.NAME,
                        BANNER_IMAGES_FORMATS.AVATARS_HOST, BANNER_IMAGES_FORMATS.MDS_GROUP_ID,
                        BANNER_IMAGES_FORMATS.NAMESPACE,
                        BANNER_IMAGES_MODERATION_VERSIONS.VERSION,
                        PRE_MODERATE_BANNERS.BID,
                        AUTO_MODERATE.BID,
                        BANNER_IMAGES.STATUS_MODERATE,
                        CLIENTS_OPTIONS.CLIENT_FLAGS)
                .from(BANNER_IMAGES)
                .join(BANNERS).on(BANNER_IMAGES.BID.eq(BANNERS.BID))
                .join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(BANNER_IMAGES_FORMATS).on(BANNER_IMAGES.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH))
                .leftJoin(CLIENTS_OPTIONS).on(CAMPAIGNS.CLIENT_ID.eq(CLIENTS_OPTIONS.CLIENT_ID))
                .leftJoin(BANNER_IMAGES_MODERATION_VERSIONS).on(BANNER_IMAGES_MODERATION_VERSIONS.BID.eq(BANNER_IMAGES.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(BANNER_IMAGES.BID.eq(PRE_MODERATE_BANNERS.BID))
                .leftJoin(AUTO_MODERATE).on(BANNER_IMAGES.BID.eq(AUTO_MODERATE.BID))
                .where(BANNER_IMAGES.IMAGE_ID.in(lockedKeys))
                .fetch(new ImageRecordMapper());
    }

    private static class ImageRecordMapper implements RecordMapper<Record, ImageWithModerationInfo> {
        @Override
        public ImageWithModerationInfo map(Record record) {
            ImageWithModerationInfo imageWithModerationInfo = new ImageWithModerationInfo()
                    .withId(record.get(BANNER_IMAGES.IMAGE_ID))
                    .withAdGroupId(record.get(BANNERS.PID))
                    .withBid(record.get(BANNER_IMAGES.BID))
                    .withCampaignId(record.get(BANNERS.CID))
                    .withClientId(record.get(CAMPAIGNS.CLIENT_ID))
                    .withUid(record.get(CAMPAIGNS.UID))
                    .withCampaignType(CampaignType.fromSource(record.get(CAMPAIGNS.TYPE)))
                    .withImageHash(record.get(BANNER_IMAGES.IMAGE_HASH))
                    .withName(record.get(BANNER_IMAGES.NAME))
                    .withBidAutoModerate(record.get(AUTO_MODERATE.BID))
                    .withBidReModerate(record.get(PRE_MODERATE_BANNERS.BID))
                    .withTransportStatus(TransportStatusAdapter.fromDb(record.get(BANNER_IMAGES.STATUS_MODERATE)))
                    .withVersion(record.get(BANNER_IMAGES_MODERATION_VERSIONS.VERSION))
                    .withClientFlags(setFromDb(record.get(CLIENTS_OPTIONS.CLIENT_FLAGS), identity()));

            if (record.get(BANNER_IMAGES_FORMATS.MDS_GROUP_ID) != null) {
                imageWithModerationInfo.withAvatarsHost(AvatarHost.fromSource(record.get(BANNER_IMAGES_FORMATS.AVATARS_HOST)))
                        .withNamespace(BannerImageFormatNamespace.fromSource(record.get(BANNER_IMAGES_FORMATS.NAMESPACE)))
                        .withMdsGroupId(record.get(BANNER_IMAGES_FORMATS.MDS_GROUP_ID).intValue());

            }

            return imageWithModerationInfo;
        }
    }

}
