package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Configuration;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalAdModerationInfo;
import ru.yandex.direct.core.entity.banner.type.internal.BannerWithInternalInfoRepositoryTypeSupport;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;

import static ru.yandex.direct.core.entity.moderation.repository.sending.ModerationRepositoryMapperProvider.createCommonBannerMapperBuilder;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
@ParametersAreNonnullByDefault
public class InternalBannerModerationRepository extends BannersModerationRepository<BannerWithInternalAdModerationInfo> {

    private final Collection<Field<?>> internalBannerFields;
    private final JooqReaderWithSupplier<BannerWithInternalAdModerationInfo> internalBannerMapper;

    public InternalBannerModerationRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(moderationOperationModeProvider);
        this.internalBannerMapper = createInternalBannerMapper();
        this.internalBannerFields = internalBannerMapper.getFieldsToRead();
    }

    @Override
    protected BannersBannerType getBannerType() {
        return BannersBannerType.internal;
    }

    @Override
    protected List<PhrasesAdgroupType> getPhrasesAdgroupTypes() {
        return List.of(PhrasesAdgroupType.internal);
    }

    @Override
    public List<BannerWithInternalAdModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                            Configuration configuration) {
        return DSL.using(configuration)
                .select(internalBannerFields)
                .from(BANNERS_INTERNAL)
                .join(BANNERS).on(BANNERS.BID.eq(BANNERS_INTERNAL.BID))
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .leftJoin(BANNER_MODERATION_VERSIONS).on(BANNER_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(PRE_MODERATE_BANNERS.BID.eq(BANNERS.BID))
                .leftJoin(AUTO_MODERATE).on(AUTO_MODERATE.BID.eq(BANNERS.BID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .where(BANNERS.BID.in(lockedKeys))
                .fetch(internalBannerMapper::fromDb);
    }

    private static JooqReaderWithSupplier<BannerWithInternalAdModerationInfo> createInternalBannerMapper() {
        return createCommonBannerMapperBuilder(BannerWithInternalAdModerationInfo::new)
                .readProperty(BannerWithInternalAdModerationInfo.VERSION, fromField(BANNER_MODERATION_VERSIONS.VERSION))
                .readProperty(BannerWithInternalAdModerationInfo.BS_BANNER_ID, fromField(BANNERS.BANNER_ID))

                .readProperty(BannerWithInternalAdModerationInfo.GEO, fromField(PHRASES.GEO)
                        .by(AdGroupMappings::geoFromDb))

                .readProperty(BannerWithInternalAdModerationInfo.DESCRIPTION, fromField(BANNERS_INTERNAL.DESCRIPTION))
                .readProperty(BannerWithInternalAdModerationInfo.TEMPLATE_ID, fromField(BANNERS_INTERNAL.TEMPLATE_ID))
                .readProperty(BannerWithInternalAdModerationInfo.TEMPLATE_VARIABLES,
                        fromField(BANNERS_INTERNAL.TEMPLATE_VARIABLES)
                                .by(BannerWithInternalInfoRepositoryTypeSupport::templateVariablesFromDb))
                .readProperty(BannerWithInternalAdModerationInfo.MODERATION_INFO,
                        fromField(BANNERS_INTERNAL.MODERATION_INFO)
                                .by(BannerWithInternalInfoRepositoryTypeSupport::moderationInfoFromDb))
                .build();
    }

}
