package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import org.jooq.Record;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.KeywordsInfoService;
import ru.yandex.direct.dbschema.ppc.Indexes;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.BIDS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_HREF_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Repository
public class KeywordsInfoRepository {

    private static final int PIDS_CHUNK_SIZE = 5_000;

    private final DslContextProvider dslContextProvider;

    @Autowired
    public KeywordsInfoRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public Map<Long, KeywordsInfoService.KeywordInfo> loadFromDb(int shard, List<Long> pids) {
        Map<Long, KeywordsInfoService.KeywordInfo> result = new HashMap<>();

        Lists.partition(pids, PIDS_CHUNK_SIZE)
                .forEach(chunk -> result.putAll(
                        loadFromDbChunked(shard, chunk)
                ));

        return result;
    }

    private Map<Long, KeywordsInfoService.KeywordInfo> loadFromDbChunked(int shard, List<Long> pids) {
        // Для подстановки параметров нам нужно какое-то одно слово, не важно какое.

        // Выбираем первое по идентификатору ключевое слово.
        // BIDS.CID нужен, потому что в таблице BIDS_HREF_PARAMS_PRIMARY, с которой мы джойнимся,
        // индекс по cid + id.
        var bidsSubQuery = dslContextProvider.ppc(shard)
                .select(BIDS.PID.as(BIDS.PID),
                        DSL.min(BIDS.CID).as(BIDS.CID),
                        DSL.min(BIDS.ID).as(BIDS.ID))
                .from(BIDS.forceIndex(Indexes.BIDS_BID_PID.getName()))
                .where(BIDS.PID.in(pids))
                .groupBy(BIDS.PID);

        // Выбираем первый по идентификатору ретаргетинг.
        var bidsRetargetingSubQuery = dslContextProvider.ppc(shard)
                .select(BIDS_RETARGETING.PID.as(BIDS_RETARGETING.PID),
                        DSL.min(BIDS_RETARGETING.RET_ID).as(BIDS_RETARGETING.RET_ID))
                .from(BIDS_RETARGETING.forceIndex(Indexes.BIDS_RETARGETING_PID.getName()))
                .where(BIDS_RETARGETING.PID.in(pids))
                .groupBy(BIDS_RETARGETING.PID);

        // Джойним все вместе, для каждой группы должно быть по одной записи.
        return dslContextProvider.ppc(shard)
                .select(PHRASES.PID,
                        BIDS_HREF_PARAMS.PARAM1,
                        BIDS_HREF_PARAMS.PARAM2,
                        BIDS.PHRASE,
                        BIDS.ID,
                        bidsRetargetingSubQuery.field(BIDS_RETARGETING.RET_ID))
                .from(PHRASES.forceIndex(Indexes.PHRASES_PRIMARY.getName()))
                .leftJoin(bidsSubQuery).on(PHRASES.PID.eq(bidsSubQuery.field(BIDS.PID)))
                .leftJoin(BIDS.forceIndex(Indexes.BIDS_PRIMARY.getName())).on(BIDS.ID.eq(bidsSubQuery.field(BIDS.ID)))
                .leftJoin(BIDS_HREF_PARAMS.forceIndex(Indexes.BIDS_HREF_PARAMS_PRIMARY.getName()))
                    .on(bidsSubQuery.field(BIDS.CID).eq(BIDS_HREF_PARAMS.CID))
                    .and(bidsSubQuery.field(BIDS.ID).eq(BIDS_HREF_PARAMS.ID))
                .leftJoin(bidsRetargetingSubQuery).on(PHRASES.PID.eq(bidsRetargetingSubQuery.field(BIDS_RETARGETING.PID)))
                .where(PHRASES.PID.in(pids))
                .fetchMap(PHRASES.PID, KeywordsInfoRepository::createKeywordInfo);
    }

    private static KeywordsInfoService.KeywordInfo createKeywordInfo(Record row) {
        return new KeywordsInfoService.KeywordInfo(
                row.get(BIDS.PHRASE),
                row.get(BIDS_HREF_PARAMS.PARAM1),
                row.get(BIDS_HREF_PARAMS.PARAM2),
                ifNotNull(row.get(BIDS.ID), Object::toString),
                ifNotNull(row.get(BIDS_RETARGETING.RET_ID), Object::toString)
        );
    }

}
