package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Record1;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.mobilecontent.model.ContentType;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentWithModerationInfo;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.dbschema.ppc.enums.MobileContentStatusiconmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.MobileContentIconModerationVersionsRecord;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.MOBILE_CONTENT_ICON;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT_ICON_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.enums.MobileContentStatusiconmoderate.Ready;
import static ru.yandex.direct.dbschema.ppc.enums.MobileContentStatusiconmoderate.Sending;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class MobileContentIconSendingRepository
        implements ModerationSendingRepository<Long, MobileContentWithModerationInfo> {
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final JooqMapperWithSupplier<MobileContentWithModerationInfo> mapper;

    public MobileContentIconSendingRepository(
            ModerationOperationModeProvider moderationOperationModeProvider) {
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        mapper = createMapper();
    }

    private static JooqMapperWithSupplier<MobileContentWithModerationInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(MobileContentWithModerationInfo::new)
                .map(property(MobileContentWithModerationInfo.ID, MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .map(property(MobileContentWithModerationInfo.STORE_CONTENT_ID, MOBILE_CONTENT.STORE_CONTENT_ID))
                .map(property(MobileContentWithModerationInfo.STORE_COUNTRY, MOBILE_CONTENT.STORE_COUNTRY))
                .map(convertibleProperty(MobileContentWithModerationInfo.OS_TYPE,
                        MOBILE_CONTENT.OS_TYPE,
                        OsType::fromSource,
                        OsType::toSource))
                .map(convertibleProperty(MobileContentWithModerationInfo.CONTENT_TYPE,
                        MOBILE_CONTENT.CONTENT_TYPE,
                        ContentType::fromSource,
                        ContentType::toSource))
                .map(property(MobileContentWithModerationInfo.BUNDLE_ID, MOBILE_CONTENT.BUNDLE_ID))
                .map(booleanProperty(MobileContentWithModerationInfo.IS_AVAILABLE, MOBILE_CONTENT.IS_AVAILABLE))
                .map(property(MobileContentWithModerationInfo.NAME, MOBILE_CONTENT.NAME))
                .map(property(MobileContentWithModerationInfo.ICON_HASH, MOBILE_CONTENT.ICON_HASH))
                .map(property(MobileContentWithModerationInfo.CLIENT_ID, MOBILE_CONTENT.CLIENT_ID))
                .map(property(MobileContentWithModerationInfo.UID, CLIENTS.CHIEF_UID))
                .map(convertibleProperty(MobileContentWithModerationInfo.TRANSPORT_STATUS,
                        MOBILE_CONTENT.STATUS_ICON_MODERATE,
                        TransportStatusAdapter::fromDb,
                        TransportStatusAdapter::toMobileContentStatusiconmoderate))
                .map(property(MobileContentWithModerationInfo.VERSION, MOBILE_CONTENT_ICON_MODERATION_VERSIONS.VERSION))
                .build();
    }

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        return configuration.dsl()
                .select(DSL.count().cast(Long.class))
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.STATUS_ICON_MODERATE.eq(Ready))
                .and(MOBILE_CONTENT.IS_AVAILABLE.eq(1L))
                .fetchOne(Record1::value1);
    }

    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration configuration) {
        return DSL.using(configuration)
                .select(MOBILE_CONTENT.MOBILE_CONTENT_ID)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(keys))
                .and(getStatusModerateCondition())
                .and(MOBILE_CONTENT.IS_AVAILABLE.eq(1L))
                .forUpdate()
                .fetch(r -> r.get(MOBILE_CONTENT.MOBILE_CONTENT_ID));
    }

    private Condition getStatusModerateCondition() {
        // В ограниченном режиме не фильтруем по статусам модерации, подходят любые
        if (moderationOperationModeProvider.getMode(MOBILE_CONTENT_ICON).equals(RESTRICTED)) {
            return DSL.trueCondition();
        }
        return MOBILE_CONTENT.STATUS_ICON_MODERATE.in(Ready, Sending);
    }

    @Override
    public List<MobileContentWithModerationInfo> loadObjectForModeration(Collection<Long> lockedIds,
                                                                         Configuration configuration) {
        return DSL.using(configuration)
                .select(mapper.getFieldsToRead())
                .from(MOBILE_CONTENT)
                .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(MOBILE_CONTENT.CLIENT_ID))
                .leftJoin(MOBILE_CONTENT_ICON_MODERATION_VERSIONS)
                .on(MOBILE_CONTENT_ICON_MODERATION_VERSIONS.MOBILE_CONTENT_ID.eq(MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(lockedIds))
                .fetch(mapper::fromDb);
    }

    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<MobileContentWithModerationInfo> loadedObjects) {
        // в ограниченном режиме не обновляем статусы модерации
        if (moderationOperationModeProvider.getMode(MOBILE_CONTENT_ICON).equals(RESTRICTED)) {
            return;
        }

        MobileContentStatusiconmoderate fromStatusModerate =
                TransportStatusAdapter.toMobileContentStatusiconmoderate(from);

        MobileContentStatusiconmoderate toStatusModerate =
                TransportStatusAdapter.toMobileContentStatusiconmoderate(to);

        DSL.using(config)
                .update(MOBILE_CONTENT)
                .set(MOBILE_CONTENT.STATUS_ICON_MODERATE, toStatusModerate)
                .where(
                        MOBILE_CONTENT.STATUS_ICON_MODERATE.eq(fromStatusModerate),
                        MOBILE_CONTENT.MOBILE_CONTENT_ID.in(
                                mapList(loadedObjects, MobileContentWithModerationInfo::getId)))
                .execute();
    }

    @Override
    public void setModerationVersions(Configuration configuration,
                                      List<Pair<MobileContentWithModerationInfo, Long>> inserts) {
        if (inserts.isEmpty()) {
            return;
        }

        InsertHelper<MobileContentIconModerationVersionsRecord> insertHelper = new InsertHelper<>(DSL.using(configuration),
                MOBILE_CONTENT_ICON_MODERATION_VERSIONS);
        inserts.forEach(insert -> {
            insertHelper.set(MOBILE_CONTENT_ICON_MODERATION_VERSIONS.MOBILE_CONTENT_ID, insert.getLeft().getId());
            insertHelper.set(MOBILE_CONTENT_ICON_MODERATION_VERSIONS.VERSION, insert.getRight());
            insertHelper.newRecord();
        });

        insertHelper
                .onDuplicateKeyUpdate()
                .set(MOBILE_CONTENT_ICON_MODERATION_VERSIONS.VERSION,
                        MySQLDSL.values(MOBILE_CONTENT_ICON_MODERATION_VERSIONS.VERSION))
                .set(MOBILE_CONTENT_ICON_MODERATION_VERSIONS.CREATE_TIME, MySQLDSL.currentLocalDateTime())
                .execute();
    }
}
