package ru.yandex.direct.core.entity.moderation.repository.sending

import org.apache.commons.lang3.tuple.Pair
import org.jooq.Condition
import org.jooq.Configuration
import org.jooq.impl.DSL
import org.jooq.impl.DSL.count
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.util.RepositoryUtils
import ru.yandex.direct.core.entity.moderation.ModerationOperationMode
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider
import ru.yandex.direct.core.entity.moderation.model.TransportStatus
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType
import ru.yandex.direct.core.entity.promoextension.model.PromoExtensionUnit
import ru.yandex.direct.core.entity.promoextension.model.PromoExtensionWithModerationInfo
import ru.yandex.direct.dbschema.ppc.Tables.CLIENTS
import ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS
import ru.yandex.direct.dbschema.ppc.Tables.PROMOACTIONS
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate.Ready
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate.Sending
import ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement
import ru.yandex.direct.jooqmapper.JooqMapperUtils.mysqlIf

@Repository
class PromoExtensionSendingRepository @Autowired constructor(
    private val moderationOperationModeProvider: ModerationOperationModeProvider,
) : ModerationSendingRepository<Long, PromoExtensionWithModerationInfo> {

    override fun getReadyObjectsCount(configuration: Configuration): Long {
        return configuration.dsl()
            .selectCount()
            .from(PROMOACTIONS)
            .where(PROMOACTIONS.STATUS_MODERATE.eq(Ready))
            .fetchOne(count())
            .toLong()
    }

    override fun lockKeys(
        ids: Collection<Long>,
        configuration: Configuration
    ): Collection<Long> {
        return DSL.using(configuration)
            .select(PROMOACTIONS.ID)
            .from(PROMOACTIONS)
            .where(PROMOACTIONS.ID.`in`(ids))
            .and(getStatusModerateCondition())
            .forUpdate()
            .fetch(PROMOACTIONS.ID)
    }

    private fun getStatusModerateCondition(): Condition {
        return if (moderationOperationModeProvider.getMode(ModerationObjectType.PROMO_EXTENSION) == ModerationOperationMode.RESTRICTED) {
            DSL.trueCondition()
        } else {
            PROMOACTIONS.STATUS_MODERATE.`in`(Ready, Sending)
        }
    }

    override fun loadObjectForModeration(
        lockedIds: Collection<Long>,
        configuration: Configuration
    ): List<PromoExtensionWithModerationInfo> {
        if (lockedIds.isEmpty()) {
            return listOf()
        }

        return DSL.using(configuration)
            .select(
                PROMOACTIONS.CLIENT_ID, CLIENTS.CHIEF_UID,
                PROMOACTIONS.MOD_VERSION, PROMOACTIONS.STATUS_MODERATE,
                PROMOACTIONS.ID, PROMOACTIONS.DESCRIPTION, PROMOACTIONS.HREF,
                PROMOACTIONS.TYPE, PROMOACTIONS.AMOUNT, PROMOACTIONS.UNIT,
                PROMOACTIONS.PREFIX, PROMOACTIONS.FINISH_DATE,
                CLIENTS_OPTIONS.CLIENT_FLAGS,
            )
            .from(PROMOACTIONS)
            .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(PROMOACTIONS.CLIENT_ID))
            .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(PROMOACTIONS.CLIENT_ID))
            .where(PROMOACTIONS.ID.`in`(lockedIds))
            .fetch()
            .map {
                PromoExtensionWithModerationInfo(
                    clientId = it.get(PROMOACTIONS.CLIENT_ID),
                    uid = it.get(CLIENTS.CHIEF_UID),
                    version = it.get(PROMOACTIONS.MOD_VERSION),
                    transportStatus = it.get(PROMOACTIONS.STATUS_MODERATE).let { status -> TransportStatusAdapter.fromDb(status) },
                    promoExtensionId = it.get(PROMOACTIONS.ID),
                    description = it.get(PROMOACTIONS.DESCRIPTION),
                    href = it.get(PROMOACTIONS.HREF),
                    type = it.get(PROMOACTIONS.TYPE),
                    amount = it.get(PROMOACTIONS.AMOUNT),
                    unit = it.get(PROMOACTIONS.UNIT)?.let{ dbValue -> PromoExtensionUnit.fromDbValue(dbValue) },
                    prefix = it.get(PROMOACTIONS.PREFIX),
                    finishDate = it.get(PROMOACTIONS.FINISH_DATE),
                    clientFlags = it.get(CLIENTS_OPTIONS.CLIENT_FLAGS).let { flags -> RepositoryUtils.setFromDb(flags) },
                )
            }
    }

    override fun updateStatusModerate(
        config: Configuration,
        from: TransportStatus,
        to: TransportStatus,
        promoExtensions: Collection<PromoExtensionWithModerationInfo>
    ) {
        // в ограниченном режиме не обновляем статусы модерации
        if (moderationOperationModeProvider.getMode(
                ModerationObjectType.PROMO_EXTENSION) == ModerationOperationMode.RESTRICTED || promoExtensions.isEmpty()) {
            return
        }
        val promoExtensionIds = promoExtensions.map { it.promoExtensionId }
        DSL.using(config)
            .update(PROMOACTIONS)
            .set(PROMOACTIONS.STATUS_MODERATE, TransportStatusAdapter.toPromoactionStatusModerate(to))
            .where(PROMOACTIONS.ID.`in`(promoExtensionIds))
            .and(PROMOACTIONS.STATUS_MODERATE.eq(TransportStatusAdapter.toPromoactionStatusModerate(from)))
            .execute()
    }

    /**
     * Реализован для полноты картины, раз объявлен в ModerationSendingRepository
     * По факту используется {@link #updateMysqlDataBeforeSending}
     */
    override fun setModerationVersions(
        configuration: Configuration,
        inserts: List<Pair<PromoExtensionWithModerationInfo, Long>>
    ) {
        if (inserts.isEmpty()) {
            return
        }
        val versionById = inserts.associate { it.left.promoExtensionId to it.right }
        DSL.using(configuration)
            .update(PROMOACTIONS)
            .set(PROMOACTIONS.MOD_VERSION, makeCaseStatement(PROMOACTIONS.ID, PROMOACTIONS.MOD_VERSION, versionById))
            .where(PROMOACTIONS.ID.`in`(versionById.keys))
            .execute()
    }

    fun updateMysqlDataBeforeSending(
        configuration: Configuration,
        updatedVersions: List<Pair<PromoExtensionWithModerationInfo, Long>>
    ) {
        if (updatedVersions.isEmpty()) {
            return
        }
        val versionById = updatedVersions.associate { it.left.promoExtensionId to it.right }

        var step = DSL.using(configuration)
            .update(PROMOACTIONS)
            .set(PROMOACTIONS.MOD_VERSION, makeCaseStatement(PROMOACTIONS.ID, PROMOACTIONS.MOD_VERSION, versionById))
        if (moderationOperationModeProvider.getMode(
                ModerationObjectType.PROMO_EXTENSION) != ModerationOperationMode.RESTRICTED) {//мб, выкинуть в DIRECT-156575
            step = step.set(PROMOACTIONS.STATUS_MODERATE,
                mysqlIf(PROMOACTIONS.STATUS_MODERATE.eq(Ready), DSL.value(Sending), PROMOACTIONS.STATUS_MODERATE))
        }
        step
            .where(PROMOACTIONS.ID.`in`(versionById.keys))
            .execute()
    }
}
