package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.google.common.collect.Sets;
import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.hrefparams.service.HrefWithParamsBuildingService;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSetWithModerationInfo;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkWithTurboHref;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatussitelinksmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerSitelinksModerationVersionsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;

import static java.util.function.Function.identity;
import static ru.yandex.direct.common.util.RepositoryUtils.setFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.GROUP_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_LINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_SET_TO_LINK;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.enums.BannersStatussitelinksmoderate.New;
import static ru.yandex.direct.dbschema.ppc.tables.BannerSitelinksModerationVersions.BANNER_SITELINKS_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

@Repository
public class SitelinksSendingRepository
        extends AssetModerationRepository<BannersRecord, BannersStatussitelinksmoderate,
        BannerSitelinksModerationVersionsRecord, SitelinkSetWithModerationInfo> {

    private static final Logger logger = LoggerFactory.getLogger(SitelinksSendingRepository.class);

    public SitelinksSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(createRepositoryParams(), moderationOperationModeProvider);
    }

    private static AssetModerationRepositoryParams<BannersRecord, BannersStatussitelinksmoderate,
            BannerSitelinksModerationVersionsRecord> createRepositoryParams() {
        return AssetModerationRepositoryParams.builder()
                .withTable(BANNERS)
                .withIdField(BANNERS.BID)
                .withStatusModerateField(BANNERS.STATUS_SITELINKS_MODERATE)
                .withTransportStatusConverter(TransportStatusAdapter::toBannersStatussitelinksmoderate)
                .withVersionsTable(BANNER_SITELINKS_MODERATION_VERSIONS)
                .withVersionsTableIdField(BANNER_SITELINKS_MODERATION_VERSIONS.BID)
                .withVersionField(BANNER_SITELINKS_MODERATION_VERSIONS.VERSION)
                .withCreateTimeField(BANNER_SITELINKS_MODERATION_VERSIONS.CREATE_TIME)
                .build();
    }

    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration configuration) {
        List<Long> allBids = DSL.using(configuration)
                .selectDistinct(BANNERS.BID)
                .from(BANNERS)
                .leftJoin(BANNER_SITELINKS_MODERATION_VERSIONS).on(BANNER_SITELINKS_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .where(BANNERS.BID.in(keys))
                .and(getStatusModerateCondition())
                .and(BANNERS.SITELINKS_SET_ID.isNotNull())
                .forUpdate()
                .fetch(r -> r.get(BANNERS.BID));

        //Заблокируем все объекты которые готовы к отправке, но отфильтруем из них валидные, не блокируя сами сеты
        //В проде есть гонка (DIRECT-117664) из-за которой  баннер может какое-то время ссылаться на несуществующий
        // SITELINKS_SET_TO_LINK
        List<Long> validBids = DSL.using(configuration)
                .selectDistinct(BANNERS.BID)
                .from(BANNERS)
                .join(SITELINKS_SET_TO_LINK).on(BANNERS.SITELINKS_SET_ID.eq(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID))
                .where(BANNERS.BID.in(allBids))
                .fetch(r -> r.get(BANNERS.BID));

        if (validBids.size() < allBids.size()) {
            Set<Long> bannersWithBrokenSitelinks = Sets.difference(Set.copyOf(allBids), Set.copyOf(validBids));
            logger.error("Some banners are pointing to removed sitelink sets, bids: {}", bannersWithBrokenSitelinks);

            configuration.dsl()
                    .update(BANNERS)
                    .set(BANNERS.STATUS_SITELINKS_MODERATE, New)
                    .set(BANNERS.SITELINKS_SET_ID, (Long) null)
                    .where(BANNERS.BID.in(bannersWithBrokenSitelinks))
                    .execute();
        }

        return validBids;
    }

    @Override
    public List<SitelinkSetWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                       Configuration configuration) {

        var selected = DSL.using(configuration)
                .select(BANNERS.BID, BANNERS.PID, BANNERS.CID, CAMPAIGNS.CLIENT_ID, CAMPAIGNS.UID,
                        BANNERS.SITELINKS_SET_ID,
                        PRE_MODERATE_BANNERS.BID, AUTO_MODERATE.BID,
                        SITELINKS_SET_TO_LINK.ORDER_NUM,
                        SITELINKS_LINKS.SL_ID, SITELINKS_LINKS.HREF, SITELINKS_LINKS.DESCRIPTION, SITELINKS_LINKS.HASH,
                        SITELINKS_LINKS.TITLE,
                        CAMP_OPTIONS.HREF_PARAMS, GROUP_PARAMS.HREF_PARAMS,
                        TURBOLANDINGS.TL_ID, TURBOLANDINGS.HREF, BANNER_SITELINKS_MODERATION_VERSIONS.VERSION,
                        BANNERS.STATUS_SITELINKS_MODERATE, CLIENTS_OPTIONS.CLIENT_FLAGS, CAMPAIGNS.TYPE
                )
                .from(BANNERS)
                .join(SITELINKS_SET_TO_LINK).on(BANNERS.SITELINKS_SET_ID.eq(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID))
                .leftJoin(SITELINKS_LINKS).on(SITELINKS_SET_TO_LINK.SL_ID.eq(SITELINKS_LINKS.SL_ID))
                .leftJoin(TURBOLANDINGS).on(TURBOLANDINGS.TL_ID.eq(SITELINKS_LINKS.TL_ID))
                .join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(CLIENTS_OPTIONS).on(CAMPAIGNS.CLIENT_ID.eq(CLIENTS_OPTIONS.CLIENT_ID))
                .leftJoin(BANNER_SITELINKS_MODERATION_VERSIONS).on(BANNER_SITELINKS_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(BANNERS.BID.eq(PRE_MODERATE_BANNERS.BID))
                .leftJoin(AUTO_MODERATE).on(BANNERS.BID.eq(AUTO_MODERATE.BID))
                .leftJoin(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .leftJoin(GROUP_PARAMS).on(BANNERS.PID.eq(GROUP_PARAMS.PID))
                .where(BANNERS.BID.in(lockedKeys))
                .orderBy(BANNERS.BID)
                .fetch();

        return convert(selected);
    }

    private <R extends Record> List<SitelinkSetWithModerationInfo> convert(Result<R> selected) {
        List<SitelinkSetWithModerationInfo> result = new ArrayList<>();

        Long prevBid = Long.MIN_VALUE;
        SitelinkSetWithModerationInfo sitelinkSetWithModerationInfo;
        List<SitelinkWithTurboHref> sitelinks = null;

        for (var row : selected) {
            if (!row.get(BANNERS.BID).equals(prevBid)) {
                sitelinks = new ArrayList<>();
                sitelinkSetWithModerationInfo = new SitelinkSetWithModerationInfo()
                        .withBid(row.get(BANNERS.BID))
                        .withAdGroupId(row.get(BANNERS.PID))
                        .withBidReModerate(row.get(PRE_MODERATE_BANNERS.BID))
                        .withBidAutoModerate(row.get(AUTO_MODERATE.BID))
                        .withCampaignId(row.get(BANNERS.CID))
                        .withClientId(row.get(CAMPAIGNS.CLIENT_ID))
                        .withUid(row.get(CAMPAIGNS.UID))
                        .withId(row.get(BANNERS.SITELINKS_SET_ID))
                        .withSitelinks(sitelinks)
                        .withTransportStatus(TransportStatusAdapter.fromDb(row.get(BANNERS.STATUS_SITELINKS_MODERATE)))
                        .withVersion(row.get(BANNER_SITELINKS_MODERATION_VERSIONS.VERSION))
                        .withClientFlags(setFromDb(row.get(CLIENTS_OPTIONS.CLIENT_FLAGS), identity()))
                        .withCampaignType(CampaignType.fromSource(row.get(CAMPAIGNS.TYPE)))
                ;

                prevBid = row.get(BANNERS.BID);

                result.add(sitelinkSetWithModerationInfo);
            }

            SitelinkWithTurboHref sitelink = new SitelinkWithTurboHref()
                    .withId(row.get(SITELINKS_LINKS.SL_ID))
                    .withHref(buildSitelinkHref(row))
                    .withTitle(row.get(SITELINKS_LINKS.TITLE))
                    .withDescription(row.get(SITELINKS_LINKS.DESCRIPTION))
                    .withOrderNum(row.get(SITELINKS_SET_TO_LINK.ORDER_NUM))
                    .withTurboLandingId(row.get(TURBOLANDINGS.TL_ID))
                    .withTurbolandingHref(row.get(TURBOLANDINGS.HREF));

            Objects.requireNonNull(sitelinks).add(sitelink);
        }

        return result;
    }

    private static String buildSitelinkHref(Record record) {
        return HrefWithParamsBuildingService.buildHrefWithParams(record.get(SITELINKS_LINKS.HREF),
                record.get(GROUP_PARAMS.HREF_PARAMS), record.get(CAMP_OPTIONS.HREF_PARAMS));
    }
}
