package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nullable;

import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.SelectJoinStep;
import org.jooq.SelectOnConditionStep;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings;
import ru.yandex.direct.core.entity.banner.model.BannerWithTextAndImageModerationInfo;
import ru.yandex.direct.core.entity.banner.model.Language;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.hrefparams.service.HrefWithParamsBuildingService;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;

import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.TEXT_AD;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES_FORMATS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.GROUP_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.enums.BannerPermalinksPermalinkAssignType.manual;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.utils.CollectionUtils.unionToSet;

@Repository
public class TextBannerModerationRepository extends BannersModerationRepository<BannerWithTextAndImageModerationInfo> {
    private final Collection<TableField<?, ?>> tableFields;
    private final OldJooqMapperWithSupplier<BannerWithTextAndImageModerationInfo> mapper;

    @Autowired
    public TextBannerModerationRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(moderationOperationModeProvider);
        this.mapper = createMapper();
        this.tableFields = unionToSet(
                mapper.getFieldsToRead(),
                List.of(CAMP_OPTIONS.HREF_PARAMS, GROUP_PARAMS.HREF_PARAMS)
        );
    }

    @Override
    protected BannersBannerType getBannerType() {
        return BannersBannerType.text;
    }

    @Override
    protected List<PhrasesAdgroupType> getPhrasesAdgroupTypes() {
        return List.of(PhrasesAdgroupType.base);
    }

    @Override
    protected Condition getStatusModerateCondition() {
        // в ограниченном режиме не фильтруем по статусам модерации, подходят любые
        if (getModerationOperationModeProvider().getMode(getModerationObjectType()).equals(RESTRICTED)) {
            return DSL.trueCondition();
        }
        return super.getStatusModerateCondition();
    }

    protected ModerationObjectType getModerationObjectType() {
        return TEXT_AD;
    }

    @Override
    public void updateStatusModerate(Configuration config, TransportStatus from, TransportStatus to,
                                     Collection<BannerWithTextAndImageModerationInfo> objects) {
        // в ограниченном режиме не обновляем статусы модерации
        if (getModerationOperationModeProvider().getMode(getModerationObjectType()).equals(RESTRICTED)) {
            return;
        }
        super.updateStatusModerate(config, from, to, objects);
    }

    @Override
    public List<BannerWithTextAndImageModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                      Configuration configuration) {
        var joinStep = DSL.using(configuration)
                .select(tableFields)
                .from(BANNERS);
        return addJoins(joinStep)
                .where(BANNERS.BID.in(lockedKeys))
                .fetch(rec -> mapper.fromDb(new BannerWithTextAndImageModerationInfo(), rec)
                        .withHref(buildBannerHref(rec))
                );
    }

    protected SelectOnConditionStep<Record> addJoins(SelectJoinStep<Record> joinStep) {
        return joinStep
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .leftJoin(AUTO_MODERATE).on(AUTO_MODERATE.BID.eq(BANNERS.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(PRE_MODERATE_BANNERS.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_MODERATION_VERSIONS).on(BANNER_MODERATION_VERSIONS.BID.eq(BANNERS.BID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .leftJoin(BANNER_TURBOLANDINGS).on(BANNER_TURBOLANDINGS.BID.eq(BANNERS.BID))
                .leftJoin(TURBOLANDINGS).on(TURBOLANDINGS.TL_ID.eq(BANNER_TURBOLANDINGS.TL_ID))
                .leftJoin(BANNER_PERMALINKS).on(BANNER_PERMALINKS.BID.eq(BANNERS.BID)
                        .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual))
                )
                .leftJoin(BANNER_IMAGES).on(BANNER_IMAGES.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_IMAGES_FORMATS).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES.IMAGE_HASH))
                .leftJoin(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .leftJoin(GROUP_PARAMS).on(PHRASES.PID.eq(GROUP_PARAMS.PID));
    }

    protected OldJooqMapperBuilder<BannerWithTextAndImageModerationInfo> buildMapper(OldJooqMapperBuilder<BannerWithTextAndImageModerationInfo> mapperBuilder) {
        return mapperBuilder
                .map(field(BANNERS.BODY, BannerWithTextAndImageModerationInfo.BODY))
                .map(field(BANNERS.BANNER_ID, BannerWithTextAndImageModerationInfo.BS_BANNER_ID))
                .map(convertibleField(CAMPAIGNS.TYPE, BannerWithTextAndImageModerationInfo.CAMPAIGN_TYPE)
                        .convertFromDbBy(CampaignType::fromSource))
                .map(field(BANNERS.TITLE, BannerWithTextAndImageModerationInfo.TITLE))
                .map(field(BANNERS.TITLE_EXTENSION, BannerWithTextAndImageModerationInfo.TITLE_EXTENSION))
                .map(field(BANNERS.BANNER_TYPE, BannerWithTextAndImageModerationInfo.BANNER_TYPE))
                .map(convertibleField(BANNERS.LANGUAGE, BannerWithTextAndImageModerationInfo.LANGUAGE)
                        .convertFromDbBy(Language::fromSource))
                .map(field(BANNER_MODERATION_VERSIONS.VERSION, BannerWithTextAndImageModerationInfo.VERSION))
                .map(convertibleField(PHRASES.GEO, BannerWithTextAndImageModerationInfo.GEO)
                        .convertToDbBy(AdGroupMappings::geoToDb)
                        .convertFromDbBy(AdGroupMappings::geoFromDb))
                .map(field(PHRASES.ADGROUP_TYPE, BannerWithTextAndImageModerationInfo.AD_GROUP_TYPE))
                .map(field(TURBOLANDINGS.HREF, BannerWithTextAndImageModerationInfo.TURBO_HREF))
                .map(convertibleField(BANNERS.STATUS_MODERATE, BannerWithTextAndImageModerationInfo.TRANSPORT_STATUS)
                        .convertToDbBy(TransportStatusAdapter::toBannerStatusModerate)
                        .convertFromDbBy(TransportStatusAdapter::fromDb))
                .map(field(BANNER_PERMALINKS.PERMALINK, BannerWithTextAndImageModerationInfo.PERMALINK))
                .map(field(BANNER_IMAGES.IMAGE_HASH, BannerWithTextAndImageModerationInfo.IMAGE_HASH))
                .map(field(BANNER_IMAGES.NAME, BannerWithTextAndImageModerationInfo.IMAGE_NAME))
                .map(convertibleField(BANNER_IMAGES.STATUS_SHOW, BannerWithTextAndImageModerationInfo.IMAGE_STATUS_SHOW)
                        .convertFromDbBy(TextBannerModerationRepository::imageStatusShowFromDb))
                .map(convertibleField(BANNER_IMAGES_FORMATS.AVATARS_HOST, BannerWithTextAndImageModerationInfo.IMAGE_AVATARS_HOST)
                        .convertFromDbBy(AvatarHost::fromSource))
                .map(convertibleField(BANNER_IMAGES_FORMATS.NAMESPACE, BannerWithTextAndImageModerationInfo.IMAGE_NAMESPACE)
                        .convertFromDbBy(BannerImageFormatNamespace::fromSource))
                .map(field(BANNER_IMAGES_FORMATS.MDS_GROUP_ID, BannerWithTextAndImageModerationInfo.IMAGE_MDS_GROUP_ID));
    }

    private static Boolean imageStatusShowFromDb(@Nullable BannerImagesStatusshow statusShow) {
        return statusShow != null ? statusShow == BannerImagesStatusshow.Yes : null;
    }

    private OldJooqMapperWithSupplier<BannerWithTextAndImageModerationInfo> createMapper() {
        var mapperBuilder = ModerationRepositoryMapperProvider.createCommonMapperBuilder(BannerWithTextAndImageModerationInfo::new);
        buildMapper(mapperBuilder);
        return mapperBuilder.build();
    }

    private static String buildBannerHref(Record rec) {
        return HrefWithParamsBuildingService.buildHrefWithParams(rec.get(BANNERS.HREF),
                rec.get(GROUP_PARAMS.HREF_PARAMS), rec.get(CAMP_OPTIONS.HREF_PARAMS));
    }
}
