package ru.yandex.direct.core.entity.moderation.repository.sending;

import java.util.Collection;
import java.util.List;

import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.RecordMapper;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingWithModerationInfo;
import ru.yandex.direct.dbschema.ppc.enums.BannerTurbolandingsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerTurboModerationVersionsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerTurbolandingsRecord;

import static java.util.function.Function.identity;
import static ru.yandex.direct.common.util.RepositoryUtils.setFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBO_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

@Repository
public class TurbolandingsSendingRepository
        extends AssetModerationRepository<BannerTurbolandingsRecord, BannerTurbolandingsStatusmoderate,
        BannerTurboModerationVersionsRecord, TurboLandingWithModerationInfo> {

    public TurbolandingsSendingRepository(ModerationOperationModeProvider moderationOperationModeProvider) {
        super(createRepositoryParams(), moderationOperationModeProvider);
    }

    private static AssetModerationRepositoryParams<BannerTurbolandingsRecord, BannerTurbolandingsStatusmoderate,
            BannerTurboModerationVersionsRecord> createRepositoryParams() {
        return AssetModerationRepositoryParams.builder()
                .withTable(BANNER_TURBOLANDINGS)
                .withIdField(BANNER_TURBOLANDINGS.BID)
                .withStatusModerateField(BANNER_TURBOLANDINGS.STATUS_MODERATE)
                .withTransportStatusConverter(TransportStatusAdapter::toBannerTurbolandingsStatusmoderate)
                .withVersionsTable(BANNER_TURBO_MODERATION_VERSIONS)
                .withVersionsTableIdField(BANNER_TURBO_MODERATION_VERSIONS.BID)
                .withVersionField(BANNER_TURBO_MODERATION_VERSIONS.VERSION)
                .withCreateTimeField(BANNER_TURBO_MODERATION_VERSIONS.CREATE_TIME)
                .build();
    }

    @Override
    public Collection<Long> lockKeys(Collection<Long> keys, Configuration config) {
        return DSL.using(config)
                .select(BANNER_TURBOLANDINGS.BID)
                .from(BANNER_TURBOLANDINGS)
                .join(BANNERS).on(BANNER_TURBOLANDINGS.BID.eq(BANNERS.BID))
                .join(PHRASES).on(BANNERS.PID.eq(PHRASES.PID))
                .leftJoin(BANNER_TURBO_MODERATION_VERSIONS)
                .on(BANNER_TURBO_MODERATION_VERSIONS.BID.eq(BANNER_TURBOLANDINGS.BID))
                .where(BANNER_TURBOLANDINGS.BID.in(keys))
                .and(PHRASES.ADGROUP_TYPE.notIn(PhrasesAdgroupType.cpm_yndx_frontpage))
                .and(getStatusModerateCondition())
                .forUpdate()
                .fetch(r -> r.get(BANNER_TURBOLANDINGS.BID));
    }

    @Override
    public long getReadyObjectsCount(Configuration configuration) {
        return configuration.dsl()
                .select(DSL.count().cast(Long.class))
                .from(BANNER_TURBOLANDINGS)
                .join(BANNERS).on(BANNER_TURBOLANDINGS.BID.eq(BANNERS.BID))
                .join(PHRASES).on(BANNERS.PID.eq(PHRASES.PID))
                .where(BANNER_TURBOLANDINGS.STATUS_MODERATE.eq(BannerTurbolandingsStatusmoderate.Ready)
                        .and(PHRASES.ADGROUP_TYPE.notIn(PhrasesAdgroupType.cpm_yndx_frontpage))
                )
                .fetchOne(Record1::value1);
    }

    @Override
    public List<TurboLandingWithModerationInfo> loadObjectForModeration(Collection<Long> lockedKeys,
                                                                        Configuration config) {
        return DSL.using(config)
                .select(BANNERS.BID, BANNERS.PID, BANNERS.CID, CAMPAIGNS.CLIENT_ID, CAMPAIGNS.UID, TURBOLANDINGS.TL_ID,
                        TURBOLANDINGS.HREF, BANNER_TURBO_MODERATION_VERSIONS.VERSION, PRE_MODERATE_BANNERS.BID,
                        AUTO_MODERATE.BID, BANNER_TURBOLANDINGS.STATUS_MODERATE, CLIENTS_OPTIONS.CLIENT_FLAGS)
                .from(TURBOLANDINGS)
                .join(BANNER_TURBOLANDINGS).on(BANNER_TURBOLANDINGS.TL_ID.eq(TURBOLANDINGS.TL_ID))
                .join(BANNERS).on(BANNER_TURBOLANDINGS.BID.eq(BANNERS.BID))
                .join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(BANNER_TURBO_MODERATION_VERSIONS).on(BANNER_TURBO_MODERATION_VERSIONS.BID.eq(BANNER_TURBOLANDINGS.BID))
                .leftJoin(PRE_MODERATE_BANNERS).on(BANNER_TURBOLANDINGS.BID.eq(PRE_MODERATE_BANNERS.BID))
                .leftJoin(AUTO_MODERATE).on(BANNER_TURBOLANDINGS.BID.eq(AUTO_MODERATE.BID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .where(BANNER_TURBOLANDINGS.BID.in(lockedKeys))
                .fetch(new TurbolandingsRecordMapper());
    }

    private static class TurbolandingsRecordMapper implements RecordMapper<Record, TurboLandingWithModerationInfo> {
        @Override
        public TurboLandingWithModerationInfo map(Record record) {
            return new TurboLandingWithModerationInfo()
                    .withId(record.get(TURBOLANDINGS.TL_ID))
                    .withAdGroupId(record.get(BANNERS.PID))
                    .withBid(record.get(BANNERS.BID))
                    .withCampaignId(record.get(BANNERS.CID))
                    .withClientId(record.get(CAMPAIGNS.CLIENT_ID))
                    .withUid(record.get(CAMPAIGNS.UID))
                    .withUrl(record.get(TURBOLANDINGS.HREF))
                    .withBidAutoModerate(record.get(AUTO_MODERATE.BID))
                    .withBidReModerate(record.get(PRE_MODERATE_BANNERS.BID))
                    .withTransportStatus(TransportStatusAdapter.fromDb(record.get(BANNER_TURBOLANDINGS.STATUS_MODERATE)))
                    .withVersion(record.get(BANNER_TURBO_MODERATION_VERSIONS.VERSION))
                    .withClientFlags(setFromDb(record.get(CLIENTS_OPTIONS.CLIENT_FLAGS), identity()));
        }
    }

}
