package ru.yandex.direct.core.entity.moderation.repository.sending.remoderation;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.moderation.model.PreModerateBanner;
import ru.yandex.direct.dbschema.ppc.tables.records.PreModerateBannersRecord;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppc.tables.AutoModerate.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.tables.PreModerateBanners.PRE_MODERATE_BANNERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class RemoderationFlagsRepositoryImpl implements RemoderationFlagsRepository {
    private static final JooqMapper<PreModerateBanner> MAPPER =
            JooqMapperWithSupplierBuilder.builder(PreModerateBanner::new)
                    .map(property(PreModerateBanner.BANNER_ID, PRE_MODERATE_BANNERS.BID))
                    .map(booleanProperty(PreModerateBanner.SEND_BANNER, PRE_MODERATE_BANNERS.SEND_BANNER))
                    .map(booleanProperty(PreModerateBanner.SEND_SITELINKS_SET, PRE_MODERATE_BANNERS.SEND_SITELINKS_SET))
                    .map(booleanProperty(PreModerateBanner.SEND_CONTACTINFO, PRE_MODERATE_BANNERS.SEND_CONTACTINFO))
                    .map(booleanProperty(PreModerateBanner.SEND_BANNER_IMAGE, PRE_MODERATE_BANNERS.SEND_BANNER_IMAGE))
                    .map(booleanProperty(PreModerateBanner.SEND_IMAGE, PRE_MODERATE_BANNERS.SEND_IMAGE))
                    .map(booleanProperty(PreModerateBanner.SEND_TURBOLANDING, PRE_MODERATE_BANNERS.SEND_TURBOLANDING))
                    .map(booleanProperty(PreModerateBanner.SEND_DISPLAY_HREFS, PRE_MODERATE_BANNERS.SEND_DISPLAY_HREFS))
                    .map(booleanProperty(PreModerateBanner.SEND_BANNER_LOGO, PRE_MODERATE_BANNERS.SEND_BANNER_LOGO))
                    .map(booleanProperty(PreModerateBanner.SEND_BANNER_BUTTON, PRE_MODERATE_BANNERS.SEND_BANNER_BUTTON))
                    .map(booleanProperty(PreModerateBanner.SEND_NAME, PRE_MODERATE_BANNERS.SEND_NAME))
                    .map(booleanProperty(PreModerateBanner.SEND_MULTICARD, PRE_MODERATE_BANNERS.SEND_MULTICARD))
                    .build();

    @Override
    public void addRemoderationFlags(Configuration configuration, Map<Long, Set<RemoderationType>> typesByBannerId) {
        InsertHelper<PreModerateBannersRecord> insertHelper = new InsertHelper<>(configuration.dsl(),
                PRE_MODERATE_BANNERS);
        typesByBannerId.forEach((bannerId, typeSet) -> {
            if (!typeSet.isEmpty()) {
                insertHelper.add(MAPPER, toPreModerateBanner(bannerId, typeSet))
                        .newRecord();
            }
        });

        insertHelper.onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    @Override
    public void removeRemoderationFlag(Configuration configuration, Collection<Long> bids, RemoderationType type) {
        configuration.dsl().update(PRE_MODERATE_BANNERS)
                .set(type.getTableFieldRef(), 0L)
                .where(PRE_MODERATE_BANNERS.BID.in(bids))
                .execute();

        List<Condition> fieldsConditions = Stream.of(RemoderationType.values())
                .map(e -> e.getTableFieldRef().eq(0L))
                .collect(Collectors.toList());

        configuration.dsl()
                .deleteFrom(PRE_MODERATE_BANNERS)
                .where(DSL.and(fieldsConditions), PRE_MODERATE_BANNERS.BID.in(bids))
                .execute();
    }

    @Override
    public void removeAutoAcceptFlag(Configuration configuration, Collection<Long> bids, AutoAcceptanceType type) {
        configuration.dsl().update(AUTO_MODERATE)
                .set(type.getTableFieldRef(), 0L)
                .where(AUTO_MODERATE.BID.in(bids))
                .execute();

        List<Condition> fieldsConditions = Stream.of(AutoAcceptanceType.values())
                .map(e -> e.getTableFieldRef().eq(0L))
                .collect(Collectors.toList());

        configuration.dsl().deleteFrom(AUTO_MODERATE)
                .where(DSL.and(fieldsConditions), AUTO_MODERATE.BID.in(bids))
                .execute();
    }

    private static PreModerateBanner toPreModerateBanner(Long bannerId, Set<RemoderationType> typeSet) {
        return new PreModerateBanner().withBannerId(bannerId)
                .withSendBanner(typeSet.contains(RemoderationType.BANNER))
                .withSendSitelinksSet(typeSet.contains(RemoderationType.SITELINKS_SET))
                .withSendContactinfo(typeSet.contains(RemoderationType.CONTACTS))
                .withSendBannerImage(typeSet.contains(RemoderationType.BANNER_IMAGE))
                .withSendImage(typeSet.contains(RemoderationType.IMAGE))
                .withSendTurbolanding(typeSet.contains(RemoderationType.TURBOLANDING))
                .withSendDisplayHrefs(typeSet.contains(RemoderationType.DISPLAY_HREFS))
                .withSendBannerLogo(typeSet.contains(RemoderationType.BANNER_LOGO))
                .withSendBannerButton(typeSet.contains(RemoderationType.BANNER_BUTTON))
                .withSendName(typeSet.contains(RemoderationType.NAME))
                .withSendMulticard(typeSet.contains(RemoderationType.MULTICARD))
                ;

    }

}
