package ru.yandex.direct.core.entity.moderation.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import one.util.streamex.StreamEx;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.MOD_EXPORT_QUEUE;

@Service
@ParametersAreNonnullByDefault
public class ModerationExportQueueService {
    private static final Logger logger = LoggerFactory.getLogger(ModerationExportQueueService.class);

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final CampaignRepository campaignRepository;

    public ModerationExportQueueService(
            ShardHelper shardHelper,
            DslContextProvider dslContextProvider,
            CampaignRepository campaignRepository
    ) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.campaignRepository = campaignRepository;
    }

    public int deleteClientCampaigns(int shard, List<ClientId> clientIds) {
        var campaignIdsToDelete = campaignRepository.getCampaignIdsByClientIds(shard, clientIds);
        if (campaignIdsToDelete.isEmpty()) {
            return 0;
        }

        logger.info(
                "These campaigns owned by clients {} will be deleted from the mod_export_queue (if they are there): {}",
                StreamEx.of(clientIds).joining(","), campaignIdsToDelete);

        var deletedCampaignsCount = dslContextProvider.ppc(shard)
                .deleteFrom(MOD_EXPORT_QUEUE)
                .where(MOD_EXPORT_QUEUE.CID.in(campaignIdsToDelete))
                .execute();

        logger.info("{} campaigns was actually deleted from mod_export_queue", deletedCampaignsCount);

        return deletedCampaignsCount;
    }

    /**
     * Переназначить поле {@link ru.yandex.direct.dbschema.ppc.tables.ModExportQueue#PAR_ID}
     */
    public void updateParIdByCampaignId(Map<Long, Long> parIdByCampaignId) {
        shardHelper.groupByShard(parIdByCampaignId.keySet(), ShardKey.CID)
                .stream()
                .forKeyValue((shard, cids) -> {
                            var uniqueCids = Set.copyOf(cids);
                            var shardedParIdByCampaignId = Maps.filterKeys(parIdByCampaignId, uniqueCids::contains);
                            var parIds = DSL.choose(MOD_EXPORT_QUEUE.CID).mapValues(shardedParIdByCampaignId);
                            dslContextProvider.ppc(shard)
                                    .update(MOD_EXPORT_QUEUE)
                                    .set(MOD_EXPORT_QUEUE.PAR_ID, parIds)
                                    .where(MOD_EXPORT_QUEUE.CID.in(cids))
                                    .execute();
                        }
                );
    }

    /**
     * @return список идентификаторов тех кампаний среди {@code campaignIds},
     * которые имеются в {@link ru.yandex.direct.dbschema.ppc.tables.ModExportQueue}
     */
    public Set<Long> remainQueuedCampaignIds(Collection<Long> campaignIds) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .stream()
                .mapKeyValue(this::remainQueuedCampaignIds)
                .flatMap(Collection::stream)
                .toSet();
    }

    private Set<Long> remainQueuedCampaignIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(MOD_EXPORT_QUEUE.CID)
                .from(MOD_EXPORT_QUEUE)
                .where(MOD_EXPORT_QUEUE.CID.in(campaignIds))
                .fetchSet(MOD_EXPORT_QUEUE.CID);
    }
}
