package ru.yandex.direct.core.entity.moderation.service;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;

import com.google.common.base.Suppliers;
import com.google.common.collect.ListMultimap;
import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithDisplayHref;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithImage;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithVcard;
import ru.yandex.direct.core.entity.banner.type.contentpromo.BannerContentPromotionRepository;
import ru.yandex.direct.core.entity.banner.type.creative.BannerCreativeRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.moderation.model.verdictrequest.contentpromotion.ContentPromotionAccessibilityData;
import ru.yandex.direct.core.entity.moderation.model.verdictrequest.contentpromotion.ContentPromotionModerationVerdictRequest;
import ru.yandex.direct.core.entity.moderation.repository.ModerationRepository;
import ru.yandex.direct.core.entity.moderation.service.sending.ContentPromotionVerdictRequestSender;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static ru.yandex.direct.common.configuration.CommonConfiguration.DIRECT_EXECUTOR_SERVICE;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@Deprecated
public class ModerationService {
    public static final int DEFAULT_PRIORITY = 50;
    public static final int DEFAULT_WIDTH = 1440;
    public static final int DEFAULT_HEIGHT = 720;

    private static final Logger LOG = LoggerFactory.getLogger(ModerationService.class);

    private final ShardHelper shardHelper;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ModerationRepository moderationRepository;
    private final BannerContentPromotionRepository bannerContentPromotionRepository;
    private final ClientRepository clientRepository;
    private final BannerCreativeRepository bannerCreativeRepository;
    private final ApplicationContext context;
    private final DslContextProvider dslContextProvider;
    private final ExecutorService executorService;
    private final Supplier<Set<ClientId>> directMonitoringClientIds;

    @Autowired
    public ModerationService(DirectConfig directConfig,
                             ShardHelper shardHelper,
                             PpcPropertiesSupport ppcPropertiesSupport,
                             ModerationRepository moderationRepository,
                             BannerContentPromotionRepository bannerContentPromotionRepository,
                             ClientRepository clientRepository,
                             BannerCreativeRepository bannerCreativeRepository,
                             ApplicationContext context,
                             DslContextProvider dslContextProvider,
                             @Qualifier(DIRECT_EXECUTOR_SERVICE) ExecutorService executorService) {
        this.shardHelper = shardHelper;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.moderationRepository = moderationRepository;
        this.bannerContentPromotionRepository = bannerContentPromotionRepository;
        this.clientRepository = clientRepository;
        this.bannerCreativeRepository = bannerCreativeRepository;
        this.context = context;
        this.dslContextProvider = dslContextProvider;
        this.executorService = executorService;

        directMonitoringClientIds = Suppliers.memoizeWithExpiration(
                this::computeDirectMonitoringClientIds,
                directConfig.getLong("direct_monitoring.cache_expire_milliseconds"),
                TimeUnit.MILLISECONDS
        );
    }

    private Set<ClientId> computeDirectMonitoringClientIds() {
        Set<Long> directMonitoringAgencyIds = ppcPropertiesSupport
                .get(PpcPropertyNames.BS_AGENCY_IDS_FOR_DIRECT_MONITORING).getOrDefault(emptySet());
        Set<ClientId> directMonitoringClientIds = ConcurrentHashMap.newKeySet();
        shardHelper.forEachShardParallel(shard -> directMonitoringClientIds.addAll(
                getSubclientIdsByAgencyClientIdsSafely(shard, directMonitoringAgencyIds)),
                executorService);
        return directMonitoringClientIds;
    }

    /**
     * Если проблемы с одним шардом, то не падаем полностью
     */
    private Set<ClientId> getSubclientIdsByAgencyClientIdsSafely(int shard, Set<Long> directMonitoringAgencyIds) {
        try {
            return clientRepository.getSubclientIdsByAgencyClientIds(shard, directMonitoringAgencyIds);
        } catch (RuntimeException ex) {
            LOG.warn("getSubclientIdsByAgencyClientIds failed in shard {}: {}", shard, ex);
        }
        return emptySet();
    }

    /**
     * Получить множество clientId, являющихся клиентами агентств Директ-Мониторинга.
     */
    public Set<ClientId> getDirectMonitoringClientIds() {
        return directMonitoringClientIds.get();
    }

    /**
     * очистка данных модерации визиток
     */
    public <B extends OldBannerWithVcard> void clearVcardsModeration(int shard, ClientId clientId,
                                                                     Collection<B> banners) {
        List<Long> bannerIds = mapList(banners, OldBanner::getId);
        Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedVcards =
                getCampaignIdToBannerIds(banners, banner -> banner.getVcardId() == null);

        clearVcardsModeration(dslContextProvider.ppc(shard), clientId, bannerIds,
                campaignIdToBannerIdsWithDeletedVcards);
    }

    /**
     * очистка данных модерации визиток
     */
    public void clearVcardsModeration(DSLContext dsl, ClientId clientId, Collection<Long> bannerIds,
                                      Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedVcards) {
        List<Long> bannerIdsToDeleteModeration =
                flatMap(campaignIdToBannerIdsWithDeletedVcards.values(), identity());
        moderationRepository.deleteBannerContactInfoVersion(dsl, bannerIdsToDeleteModeration);
        moderationRepository.deleteBannerContactInfoModReason(dsl, bannerIds);
    }

    /**
     * очистка данных модерации сайтлинк сетов
     */
    public <B extends OldBannerWithSitelinks> void clearSitelinksSetsModeration(int shard, ClientId clientId,
                                                                                Collection<B> banners) {
        List<Long> bannerIds = mapList(banners, OldBanner::getId);
        Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedSitelinks =
                getCampaignIdToBannerIds(banners, banner -> banner.getSitelinksSetId() == null);

        clearSitelinksSetsModeration(dslContextProvider.ppc(shard), clientId, bannerIds,
                campaignIdToBannerIdsWithDeletedSitelinks);
    }

    /**
     * очистка данных модерации сайтлинк сетов
     */
    public void clearSitelinksSetsModeration(DSLContext dsl, ClientId clientId, Collection<Long> bannerIds,
                                             Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedSitelinks) {
        List<Long> bannerIdsToDeleteModeration =
                flatMap(campaignIdToBannerIdsWithDeletedSitelinks.values(), identity());
        moderationRepository.deleteBannerSitelinkSetVersion(dsl, bannerIdsToDeleteModeration);
        moderationRepository.deleteBannerSitelinkSetModReason(dsl, bannerIds);
    }

    /**
     * очистка данных модерации изображений баннеров
     */
    public <B extends OldBannerWithBannerImage> void clearBannerImagesModeration(int shard, ClientId clientId,
                                                                                 Collection<B> banners) {
        //уберем из модерации удаленные
        Map<Long, List<Long>> campaignIdToBannerIds =
                getCampaignIdToBannerIds(banners, banner -> banner.getBannerImage() == null);
        List<Long> bannerIds = mapList(banners, OldBanner::getId);

        clearImagesModeration(dslContextProvider.ppc(shard), clientId, bannerIds, campaignIdToBannerIds);
    }

    /**
     * очистка данных модерации изображений ГО
     */
    public <B extends OldBannerWithImage> void clearImagesModeration(int shard, ClientId clientId,
                                                                     Collection<B> banners) {
        //уберем из модерации удаленные
        Map<Long, List<Long>> campaignIdToBannerIds =
                getCampaignIdToBannerIds(banners, banner -> banner.getImage() == null);
        List<Long> bannerIds = mapList(banners, OldBanner::getId);

        clearImagesModeration(dslContextProvider.ppc(shard), clientId, bannerIds, campaignIdToBannerIds);
    }

    /**
     * очистка данных модерации изображений
     */
    public void clearImagesModeration(DSLContext dsl,
                                      ClientId clientId,
                                      List<Long> bannerIds,
                                      Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedImages) {

        List<Long> bannerIdsToDeleteModeration = flatMap(campaignIdToBannerIdsWithDeletedImages.values(), identity());
        moderationRepository.deleteBannerImageVersion(dsl, bannerIdsToDeleteModeration);
        moderationRepository.deleteBannerImageModReason(dsl, bannerIds);
    }

    /**
     * очистка данных модерации отображаемых ссылок
     */
    public <B extends OldBannerWithDisplayHref> void clearDisplayHrefsModeration(int shard, ClientId clientId,
                                                                                 Collection<B> banners) {
        Map<Long, List<Long>> campaignIdToBannerIds =
                getCampaignIdToBannerIds(banners, banner -> banner.getDisplayHref() == null);
        List<Long> bannerIds = mapList(banners, OldBanner::getId);

        clearDisplayHrefsModeration(dslContextProvider.ppc(shard), clientId, bannerIds, campaignIdToBannerIds);
    }

    public void clearDisplayHrefsModeration(DSLContext dsl, ClientId clientId, Collection<Long> bannerIds,
                                            Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedDisplayHrefs) {
        List<Long> bannerIdsToDeleteModeration =
                flatMap(campaignIdToBannerIdsWithDeletedDisplayHrefs.values(), identity());

        moderationRepository.deleteBannerDisplayHrefVersion(dsl, bannerIdsToDeleteModeration);
        moderationRepository.deleteBannerDisplayHrefModReason(dsl, bannerIds);
    }

    public void clearVideoAdditionsModeration(DSLContext dsl,
                                              ClientId clientId,
                                              List<Long> bannerIds,
                                              Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedCreatives) {

        List<Long> bannerIdsToDeleteModeration = flatMap(campaignIdToBannerIdsWithDeletedCreatives.values(),
                identity());
        moderationRepository.deleteBannerVideoAdditionVersion(dsl, bannerIdsToDeleteModeration);
        moderationRepository.deleteBannerVideoAdditionModReason(dsl, bannerIds);

    }

    /**
     * очистка данных модерации турболендингов
     */
    public <B extends OldBannerWithTurboLanding> void clearBannerTurboLandingsModeration(int shard, ClientId clientId,
                                                                                         Collection<B> banners) {
        List<Long> bannerIds = mapList(banners, B::getId);
        //уберем из модерации удаленные
        Map<Long, List<Long>> campaignIdToBannerIds =
                getCampaignIdToBannerIds(banners, banner -> banner.getTurboLandingId() == null);
        clearBannerTurboLandingsModeration(dslContextProvider.ppc(shard), bannerIds, campaignIdToBannerIds);
    }

    /**
     * очистка данных модерации турболендингов
     */
    public void clearBannerTurboLandingsModeration(
            DSLContext dsl,
            List<Long> allBannerIds,
            Map<Long, List<Long>> campaignIdToBannerIdsForDelete) {
        List<Long> bannerIdsToDeleteModeration = flatMap(campaignIdToBannerIdsForDelete.values(), identity());
        moderationRepository.deleteBannerTurboLandingVersion(dsl, bannerIdsToDeleteModeration);

        moderationRepository.deleteBannerTurboLandingModReason(dsl, allBannerIds);
    }

    /**
     * Удаление флагов модерации баннера
     */
    public void deleteBannerModerationFlags(int shard, List<OldBanner> banners) {
        List<Long> bannerIds = mapList(banners, OldBanner::getId);
        deleteBannerModerationFlags(dslContextProvider.ppc(shard), bannerIds);
    }

    /**
     * Удаление флагов модерации баннера
     */
    public void deleteBannerModerationFlags(DSLContext dslContext, List<Long> bannerIds) {
        moderationRepository.deletePostModerate(dslContext.configuration(), bannerIds);
        moderationRepository.deleteAutoModerate(dslContext.configuration(), bannerIds);
        moderationRepository.deleteBannerModEdit(dslContext, bannerIds);
    }

    /**
     * Удаление флагов модерации баннера
     */
    public void deleteBannerPostAndAuto(Configuration configuration, Collection<Long> bannerIds) {
        moderationRepository.deletePostModerate(configuration, bannerIds);
        moderationRepository.deleteAutoModerate(configuration, bannerIds);
    }

    /**
     * Удаление информации о заданных группах из модерации.
     *
     * @param shard                 шард
     * @param clientId              идентификатор клиента
     * @param campaignIdToAdGroupId мапа campaignId -> список adGroupId
     */
    public void deleteAdGroups(int shard, ClientId clientId, ListMultimap<Long, Long> campaignIdToAdGroupId) {
        if (campaignIdToAdGroupId.isEmpty()) {
            return;
        }

        moderationRepository.deleteAdGroupModReason(shard, campaignIdToAdGroupId.values());
        moderationRepository.deleteAdGroupVersion(shard, campaignIdToAdGroupId.values());
    }

    private <B extends OldBanner> Map<Long, List<Long>> getCampaignIdToBannerIds(Collection<B> banners,
                                                                                 Predicate<B> predicate) {
        return StreamEx.of(banners)
                .filter(predicate)
                .mapToEntry(OldBanner::getCampaignId, OldBanner::getId)
                .grouping();
    }

    /**
     * Отправляет запросы в модерацию для продвижения контента, если доступность контента изменилась (например, если
     * контент был доступен и стал недоступен или наоборот).
     *
     * @param shard                             шард
     * @param contentPromotionAccessibilityData данные о доступности контента
     * @param sender                            callback для отправки запросов в модерацию
     */
    public void makeContentPromotionAccessibilityVerdictsRequests(
            int shard,
            List<ContentPromotionAccessibilityData> contentPromotionAccessibilityData,
            Consumer<List<ContentPromotionModerationVerdictRequest>> sender) {
        if (contentPromotionAccessibilityData.isEmpty()) {
            return;
        }

        Map<Long, List<Long>> beenInModerationBannerIdsByContentId =
                bannerContentPromotionRepository.getBeenInModerationBannerIdsByContentId(shard,
                        mapList(contentPromotionAccessibilityData,
                                ContentPromotionAccessibilityData::getContentPromotionId));

        Map<Long, Boolean> isContentPromotionInaccessibleByBannerId =
                new HashMap<>(contentPromotionAccessibilityData.size());

        contentPromotionAccessibilityData.forEach(accessibilityData -> {
            Boolean isInaccessible = accessibilityData.getIsInaccessible();

            beenInModerationBannerIdsByContentId.getOrDefault(accessibilityData.getContentPromotionId(), emptyList())
                    // Для одного баннера может быть несколько записей в очереди, выбираем последнюю
                    .forEach(bannerId -> isContentPromotionInaccessibleByBannerId.put(bannerId, isInaccessible));
        });

        ContentPromotionVerdictRequestSender contentPromotionVerdictRequestSender =
                context.getBean(ContentPromotionVerdictRequestSender.class);

        contentPromotionVerdictRequestSender
                .setIsContentPromotionInaccessibleByBannerId(isContentPromotionInaccessibleByBannerId);

        contentPromotionVerdictRequestSender
                .send(shard, isContentPromotionInaccessibleByBannerId.keySet(),
                        (o) -> System.currentTimeMillis(),
                        (o) -> null,
                        sender);
    }
}
