package ru.yandex.direct.core.entity.moderation.service.contentpromotion.subtype;

import java.io.IOException;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithModerationInfo;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.moderation.model.AspectRatio;
import ru.yandex.direct.core.entity.moderation.model.BannerLink;
import ru.yandex.direct.core.entity.moderation.model.contentpromotion.ContentPromotionBannerRequestData;
import ru.yandex.direct.core.entity.moderation.model.contentpromotion.ContentPromotionVideoBannerRequestData;
import ru.yandex.direct.libs.video.VideoClientUtils;
import ru.yandex.direct.libs.video.model.VideoBanner;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@Component
@ParametersAreNonnullByDefault
public class ContentPromotionVideoModerationTypeSupport implements ContentPromotionModerationTypeSupport {

    private static final Logger logger = LoggerFactory.getLogger(ContentPromotionVideoModerationTypeSupport.class);
    private final BannersUrlHelper bannersUrlHelper;

    public ContentPromotionVideoModerationTypeSupport(
            BannersUrlHelper bannersUrlHelper) {
        this.bannersUrlHelper = bannersUrlHelper;
    }

    @Override
    public ContentPromotionContentType getContentPromotionType() {
        return ContentPromotionContentType.VIDEO;
    }

    @Override
    public ContentPromotionBannerRequestData buildModerationData(BannerWithModerationInfo banner) {
        ContentPromotionVideoBannerRequestData data = new ContentPromotionVideoBannerRequestData();

        String visitUrl = banner.getContentPromotionVisitUrl();
        data.setLinks(singletonList(
                new BannerLink()
                        .setHref(visitUrl)));
        data.setDomain(bannersUrlHelper.extractHostFromHrefWithWwwOrNull(visitUrl));

        VideoBanner videoBanner = fromJson(banner.getContentPromotionMetadata(), VideoBanner.class);
        data.setLogin(banner.getLogin());
        data.setTitle(banner.getTitle());
        data.setBody(banner.getBody());

        data.setVideoHostingUrl(VideoClientUtils.getUniformUrl(videoBanner));
        String previewUrl = banner.getContentPromotionPreviewUrl();
        data.setCreativePreviewUrl(previewUrl.startsWith("//") ? "https:" + previewUrl : previewUrl);

        data.setCreativeId(banner.getContentPromotionId());

        String mediaDuration = videoBanner.getMediaDuration();
        data.setDuration(mediaDuration != null ? Long.parseLong(mediaDuration) : null);
        data.setHost(videoBanner.getHost());

        try {
            JsonNode node = new ObjectMapper().readTree(videoBanner.getThAttrs());
            Long width = node.get("OriginalWidth").asLong();
            Long height = node.get("OriginalHeight").asLong();
            data.setAspectRatio(new AspectRatio(width, height));
        } catch (IOException e) {
            logger.info("Can't get aspect ratio for video with id: {}", banner.getContentPromotionId());
        }

        return data;
    }
}
