package ru.yandex.direct.core.entity.moderation.service.receiving;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record;
import org.jooq.TableField;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_BUTTONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_LOGOS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;

@ParametersAreNonnullByDefault
@Component
public class BannerChangesValidator {
    private static final List<TableField<BannersRecord, ?>> ALLOWED_FIELDS = List.of(
            BANNERS.STATUS_MODERATE, BANNERS.STATUS_POST_MODERATE, BANNERS.STATUS_BS_SYNCED, BANNERS.STATUS_SHOW,
            BANNERS.FLAGS, BANNERS.LANGUAGE, BANNERS.STATUS_SITELINKS_MODERATE
    );

    public boolean validateChanges(BulkUpdate.RowChanges<? extends Record> changes) {
        var changedFields = changes.getNewColumnValues().keySet();

        if (changedFields.stream().allMatch(field -> field.getTable() == PHRASES)) {
            // Изменения в группах не проверяем (пока меняется только StatusBsSynced)
            return true;
        }

        if (changedFields.stream().allMatch(field -> field.getTable() == BANNER_TURBOLANDINGS)) {
            // Изменения в турболендингах не проверяем (пока меняется только StatusModerate)
            return true;
        }

        if (changedFields.stream().allMatch(field -> field.getTable() == BANNER_IMAGES)) {
            // Изменения в картинках не проверяем (пока меняется только StatusModerate)
            return true;
        }

        if (changedFields.stream().allMatch(field -> field.getTable() == IMAGES)) {
            // Изменения в картинках ГО не проверяем (пока меняется только StatusModerate и extracted_text)
            return true;
        }

        if (changedFields.stream().allMatch(field -> field.getTable() == BANNER_DISPLAY_HREFS)) {
            // Изменения в отображаемых урлах не проверяем (пока меняется только StatusModerate)
            return true;
        }

        if (changedFields.stream().allMatch(field -> field.getTable() == BANNER_BUTTONS)) {
            // Изменения в кнопках не проверяем (пока меняется только StatusModerate)
            return true;
        }

        if (changedFields.stream().allMatch(field -> field.getTable() == BANNER_LOGOS)) {
            // Изменения в логотипах не проверяем (пока меняется только StatusModerate)
            return true;
        }

        if (changedFields.stream().allMatch(field -> field.getTable() == BANNERS_PERFORMANCE)) {
            // Изменения в связках баннер-креатив пока не проверяем
            // (пока меняется только StatusModerate и extracted_text)
            return true;
        }

        if (changedFields.stream().allMatch(field -> field.getTable() == PERF_CREATIVES)) {
            // Изменения в баннерстораджных креативах пока не проверяем
            // (пока меняется только StatusModerate и additional_data)
            return true;
        }

        if (changedFields.stream().allMatch(field -> field.getTable() == BANNERS)) {
            // Проверяем изменения в баннерах
            return validateBannerChanges(changes);
        }

        return false;
    }

    private boolean validateBannerChanges(BulkUpdate.RowChanges<? extends Record> changes) {
        var newValues = changes.getNewColumnValues();
        // Ограничиваем список полей, которые можно изменить у баннера
        if (!ALLOWED_FIELDS.containsAll(newValues.keySet())) {
            return false;
        }

        // Если баннер приняли, то statusPostmoderate должен быть Yes
        if (newValues.get(BANNERS.STATUS_MODERATE) == BannersStatusmoderate.Yes &&
                newValues.get(BANNERS.STATUS_POST_MODERATE) != BannersStatuspostmoderate.Yes) {
            return false;
        }

        // Если баннер отклонили, то statusPostmoderate должен быть No
        if (newValues.get(BANNERS.STATUS_MODERATE) == BannersStatusmoderate.No &&
                newValues.get(BANNERS.STATUS_POST_MODERATE) == BannersStatuspostmoderate.Yes) {
            return false;
        }

        // После модерации баннер всегда должен переотправляться в БК
        if (newValues.get(BANNERS.STATUS_BS_SYNCED) != BannersStatusbssynced.No) {
            return false;
        }
        // Транспорт не должен включать показы никогда, даже если баннер приняли на модерации
        if (newValues.get(BANNERS.STATUS_SHOW) == BannersStatusshow.Yes) {
            return false;
        }

        return true;
    }
}
