package ru.yandex.direct.core.entity.moderation.service.receiving;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.jooq.Configuration;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationResponse;
import ru.yandex.direct.core.entity.moderation.repository.receiving.BannersModerationReceivingRepository;
import ru.yandex.direct.core.entity.moderation.service.receiving.processing_configurations.BannerOperations;
import ru.yandex.direct.core.entity.moderation.service.receiving.processor.response_parser.BannerResponseParser;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

public abstract class BannerModerationResponsesReceivingService extends BaseModerationReceivingService<BannerModerationResponse, Long> {
    private static final Logger logger = LoggerFactory.getLogger(BannerModerationResponsesReceivingService.class);

    protected final BannerRelationsRepository bannerRelationsRepository;
    protected final AdGroupRepository adGroupRepository;

    public BannerModerationResponsesReceivingService(
            BannerRelationsRepository bannerRelationsRepository,
            AdGroupRepository adGroupRepository,
            DslContextProvider dslContextProvider,
            BannersModerationReceivingRepository bannersModerationReceivingRepository,
            BannerOperations bannerOperations,
            BannerResponseParser bannerResponseParser,
            BannerChangesValidator bannerChangesValidator,
            PpcPropertiesSupport ppcPropertiesSupport) {
        super(dslContextProvider, bannersModerationReceivingRepository, bannerResponseParser, bannerOperations,
                ppcPropertiesSupport);

        getBulkUpdateHolder().setValueMerger(BannerModerationResponsesReceivingService::bulkUpdateFieldMerge);

        // Пока лишь записываем в лог те изменения, которые не проходят валидацию. Планируем через месяц посмотреть,
        // что налоггировалось, и потом уже включить полноценную валидацию. Задача про это: DIRECT-121320
        getBulkUpdateHolder().setValidator(changes -> {
            if (!bannerChangesValidator.validateChanges(changes)) {
                logger.error("Attempt to execute incorrect update: {}", changes);
            }
            return true;
        });

        this.bannerRelationsRepository = bannerRelationsRepository;
        this.adGroupRepository = adGroupRepository;
    }

    private static Object bulkUpdateFieldMerge(Object f1, Object f2) {
        if (f1 instanceof BannersStatuspostmoderate && (f1 == BannersStatuspostmoderate.Rejected || f2 == BannersStatuspostmoderate.Rejected)) {
            return BannersStatuspostmoderate.Rejected;
        } else if ((f1 instanceof BannersStatusmoderate && (f1 == BannersStatusmoderate.No || f2 == BannersStatusmoderate.No))) {
            return BannersStatusmoderate.No;
        } else {
            throw new IllegalStateException("Can't merge " + f1 + " " + f2);
        }
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    // Костыль, нужен до выполнения тикета
    // DIRECT-111278: На стороне модерации у некоторых типов объектов отбрасывается campaignId из входящих запросов
    @Override
    protected void preProcessLocked(List<BannerModerationResponse> responses, Configuration configuration) {
        super.preProcessLocked(responses, configuration);

        Map<Long, BannerModerationResponse> responsesWithoutCampaignId =
                responses.stream()
                        .filter(r -> r.getMeta().getCampaignId() == 0)
                        .collect(Collectors.toMap(r -> r.getMeta().getBannerId(), Function.identity()));

        if (responsesWithoutCampaignId.isEmpty()) {
            return;
        }

        Map<Long, Long> campaignIds = bannerRelationsRepository.getCampaignIdsByBannerIds(DSL.using(configuration),
                responsesWithoutCampaignId.keySet());

        for (Map.Entry<Long, Long> bid2cid : campaignIds.entrySet()) {
            responsesWithoutCampaignId.get(bid2cid.getKey()).getMeta().setCampaignId(bid2cid.getValue());
        }
    }

    @Override
    protected String getName() {
        return "banners";
    }
}
