package ru.yandex.direct.core.entity.moderation.service.receiving;

import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

import ru.yandex.direct.bannerstorage.client.BannerStorageClient;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.moderation.model.ModerationDecision;
import ru.yandex.direct.core.entity.moderation.model.bannerstorage.BannerstorageCreativeModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.bannerstorage.BannerstorageCreativeModerationResponse;
import ru.yandex.direct.core.entity.moderation.repository.receiving.BannerstorageCreativesModerationReceivingRepository;
import ru.yandex.direct.core.entity.moderation.service.receiving.processing_configurations.BannerstorageCreativesOperations;
import ru.yandex.direct.core.entity.moderation.service.receiving.processor.ChooseResponseWithMaxVersion;
import ru.yandex.direct.core.entity.moderation.service.receiving.processor.response_parser.BannerstorageCreativeResponseParser;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.utils.ThreadUtils;

import static org.springframework.beans.factory.config.ConfigurableBeanFactory.SCOPE_PROTOTYPE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Service
@Scope(SCOPE_PROTOTYPE)
public class BannerstorageCreativesModerationResponseReceivingService extends
        BaseModerationReceivingService<BannerstorageCreativeModerationResponse, Long> {

    private static final Logger logger =
            LoggerFactory.getLogger(BannerstorageCreativesModerationResponseReceivingService.class);
    private final BannerStorageClient bannerStorageClient;
    private final CreativeRepository creativeRepository;

    @Autowired
    protected BannerstorageCreativesModerationResponseReceivingService(
            DslContextProvider ppcDslContextProvider,
            BannerstorageCreativesModerationReceivingRepository moderationReceivingRepository,
            BannerstorageCreativeResponseParser moderationResponseParser,
            BannerstorageCreativesOperations bannerstorageCreativesOperations,
            BannerStorageClient bannerStorageClient,
            CreativeRepository creativeRepository,
            PpcPropertiesSupport ppcPropertiesSupport) {
        super(ppcDslContextProvider, moderationReceivingRepository, moderationResponseParser,
                bannerstorageCreativesOperations,
                // Для креативов bannerstorage специальная логика с критичными причинами пока не поддерживается
                ChooseResponseWithMaxVersion.INSTANCE,
                ppcPropertiesSupport);
        this.bannerStorageClient = bannerStorageClient;
        this.creativeRepository = creativeRepository;
    }

    @Override
    protected void preProcessResponses(int shard, List<BannerstorageCreativeModerationResponse> responses) {
        super.preProcessResponses(shard, responses);

        // Определим, какие из креативов привязаны к существующим баннерам
        // Для тех, которые не привязаны (баннер можно удалить пока креатив модерируется),
        // не зовём ручку в BannerStorage, чтобы статусы не расходились: в базе Директа вердикт не применится
        // (т.к. не будет найдена соответствующая версия через связку
        // perf_creatives + banners_performance + bannerstorage_creatives_moderation_versions)
        Set<Long> linkedWithBanners = creativeRepository.selectCreativesLinkedWithBanners(
                shard, mapList(responses, r -> r.getMeta().getCreativeId()));

        // Позвать bannerstorage для указанных версий креативов
        // чтобы синхронизировать статусы креативов и стриггерить отправку в RTB Host
        // Делаем перед изменениями в базе, т.к. важно, чтобы этот шаг не потерялся
        for (BannerstorageCreativeModerationResponse response : responses) {
            BannerstorageCreativeModerationMeta meta = response.getMeta();
            if (!linkedWithBanners.contains(meta.getCreativeId())) {
                logger.info("Skip verdict for creativeId={}: banner is deleted", linkedWithBanners);
            } else {
                if (response.getResult().getVerdict() == ModerationDecision.Yes) {
                    logger.info("Approving creative {} version {}", meta.getCreativeId(), meta.getCreativeVersionId());
                    ThreadUtils.execWithRetries(attempt -> {
                        bannerStorageClient.approveExternallyModeratedCreative(
                                (int) meta.getCreativeId(),
                                (int) meta.getCreativeVersionId()
                        );
                        logger.info("Approved creative {} version {}", meta.getCreativeId(),
                                meta.getCreativeVersionId());
                    }, 5, 5000, 1.2, logger);
                } else if (response.getResult().getVerdict() == ModerationDecision.No) {
                    logger.info("Rejecting creative {} version {}", meta.getCreativeId(), meta.getCreativeVersionId());
                    ThreadUtils.execWithRetries(attempt -> {
                        bannerStorageClient.rejectExternallyModeratedCreative(
                                (int) meta.getCreativeId(),
                                (int) meta.getCreativeVersionId()
                        );
                        logger.info("Rejected creative {} version {}", meta.getCreativeId(),
                                meta.getCreativeVersionId());
                    }, 5, 5000, 1.2, logger);
                } else {
                    logger.info("Unexpected verdict found in moderation response: {}", toJson(response));
                    throw new UnsupportedOperationException("Unexpected verdict found");
                }
            }
        }
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected String getName() {
        return "bannerstorage-creatives";
    }
}
