package ru.yandex.direct.core.entity.moderation.service.receiving;


import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Configuration;
import org.jooq.Record;
import org.slf4j.Logger;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResultResponse;
import ru.yandex.direct.core.entity.moderation.model.ModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationResult;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderation.repository.receiving.ModerationReceivingRepository;
import ru.yandex.direct.core.entity.moderation.service.receiving.processing_configurations.ResponseOperationsChain;
import ru.yandex.direct.core.entity.moderation.service.receiving.processor.ChooseResponseWithMaxVersion;
import ru.yandex.direct.core.entity.moderation.service.receiving.processor.ModerateResponseChooser;
import ru.yandex.direct.core.entity.moderation.service.receiving.processor.ModerationResponseParser;
import ru.yandex.direct.core.entity.moderation.service.receiving.processor.ModerationResponseProcessingResult;
import ru.yandex.direct.core.entity.moderation.service.receiving.processor.ModerationResponsesProcessingMachine;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.collections4.CollectionUtils.subtract;

public abstract class BaseModerationReceivingService<T extends AbstractModerationResultResponse<?
        extends ModerationMeta, ? extends ModerationResult>, U extends Comparable<U>> implements ModerationReceivingService<T> {

    private final DslContextProvider ppcDslContextProvider;
    private final ModerationResponsesProcessingMachine<T, U> responseProcessor;
    private final ModerationReceivingRepository<? extends Record, ? extends
            Enum<?>, U> moderationReceivingRepository;
    private final ModerationResponseParser<T, U> moderationResponseParser;
    private final ResponseOperationsChain<T> moderationResponseProcessingOps;
    private final BulkUpdateHolder bulkUpdateHolder;
    private final ModerateResponseChooser moderateResponseChooser;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    protected BaseModerationReceivingService(DslContextProvider ppcDslContextProvider,
                                             ModerationReceivingRepository<? extends Record, ? extends
                                                     Enum<?>, U> moderationReceivingRepository,
                                             ModerationResponseParser<T, U> moderationResponseParser,
                                             ResponseOperationsChain<T> moderationResponseProcessingOps,
                                             PpcPropertiesSupport ppcPropertiesSupport) {
        this(ppcDslContextProvider, moderationReceivingRepository, moderationResponseParser,
                moderationResponseProcessingOps,
                ChooseResponseWithMaxVersion.INSTANCE,
                ppcPropertiesSupport);
    }

    protected BaseModerationReceivingService(DslContextProvider ppcDslContextProvider,
                                             ModerationReceivingRepository<? extends Record, ? extends
                                                     Enum<?>, U> moderationReceivingRepository,
                                             ModerationResponseParser<T, U> moderationResponseParser,
                                             ResponseOperationsChain<T> moderationResponseProcessingOps,
                                             ModerateResponseChooser moderateResponseChooser,
                                             PpcPropertiesSupport ppcPropertiesSupport) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.moderationReceivingRepository = moderationReceivingRepository;
        this.moderationResponseParser = moderationResponseParser;
        this.moderationResponseProcessingOps = moderationResponseProcessingOps;
        this.bulkUpdateHolder = new BulkUpdateHolder();
        this.moderateResponseChooser = moderateResponseChooser;
        this.ppcPropertiesSupport = ppcPropertiesSupport;

        this.responseProcessor = buildProcessor();
    }

    protected ModerationResponsesProcessingMachine<T, U> buildProcessor() {
        return ModerationResponsesProcessingMachine.<T, U>builder()
                .withResponseParser(moderationResponseParser)
                .withModerateResponseChooser(moderateResponseChooser)
                .withObjectLocker(this::lockIds)
                .withOperations(moderationResponseProcessingOps)
                .withResponsesConsumer(this::preProcessLocked)
                .withBulkUpdateHolder(bulkUpdateHolder)
                .build();
    }

    public BulkUpdateHolder getBulkUpdateHolder() {
        return bulkUpdateHolder;
    }

    protected List<U> lockIdWithVersion(Configuration configuration,
                                        Collection<ModeratedObjectKeyWithVersion<U>> ids) {
        return moderationReceivingRepository.selectByKeysAndExportVersionsWithLock(configuration, ids);
    }

    protected abstract Logger getLogger();

    protected void preProcessLocked(List<T> responses, Configuration configuration) {
    }

    private List<U> lockIds(Configuration configuration, Collection<ModeratedObjectKeyWithVersion<U>> responses) {

        List<U> lockedIds = lockIdWithVersion(configuration, responses);

        if (lockedIds.size() != responses.size()) {
            Collection<U> notLockedObjects = subtract(
                    responses.stream().map(ModeratedObjectKeyWithVersion::getKey).collect(toList()),
                    lockedIds
            );

            getLogger().warn("Fail to lock {} for moderation response! {}", getName(),
                    notLockedObjects.stream().map(Object::toString).collect(joining(", ")));
        }

        return lockedIds;
    }

    protected abstract String getName();

    private List<T> processResponsesInTransaction(Configuration configuration, List<T> responses,
                                                  int[] unknownVerdictCount) {

        ModerationResponseProcessingResult<T> responseProcessingResult = responseProcessor.processResponses(responses,
                configuration);

        unknownVerdictCount[0] = responseProcessingResult.getUnknownResponses();

        return responseProcessingResult.getSuccessfulResponses();
    }

    protected void preProcessResponses(int shard, List<T> responses) {
    }

    protected void postProcessResponses(int shard, List<T> responses) {
    }

    @Override
    public Pair<Integer, List<T>> processModerationResponses(int shard, List<T> responses) {
        if (responses.isEmpty()) {
            return Pair.of(0, responses);
        }

        AtomicReference<List<T>> result = new AtomicReference<>(new ArrayList<>());
        int[] unknownVerdictCount = {0};

        preProcessResponses(shard, responses);

        ppcDslContextProvider
                .ppcTransaction(shard,
                        configuration -> result
                                .set(processResponsesInTransaction(configuration, responses, unknownVerdictCount)));

        postProcessResponses(shard, responses);

        return Pair.of(unknownVerdictCount[0], result.get());
    }

    public static class ModeratedObjectKeyWithVersion<K> {
        public static final int ANY_VERSION = -1;
        private final K key;
        private final long versionId;

        public ModeratedObjectKeyWithVersion(K key, long versionId) {
            this.key = key;
            this.versionId = versionId;
        }

        public K getKey() {
            return key;
        }

        public long getVersionId() {
            return versionId;
        }
    }
}
