package ru.yandex.direct.core.entity.moderation.service.receiving.dyn_disclaimers;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Record2;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static org.jooq.impl.DSL.val;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDITIONS_ITEM_DISCLAIMERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_ADDITIONS;
import static ru.yandex.direct.dbschema.ppc.enums.BannersAdditionsAdditionsType.disclaimer;

@Repository
public class DisclaimersRepository {
    private final ShardHelper shardHelper;

    @Autowired
    public DisclaimersRepository(ShardHelper shardHelper) {
        this.shardHelper = shardHelper;
    }

    public void removeBannerLinks(Configuration configuration, Map<DisclaimerUpdateRequest, Long> requestsWithId) {
        List<Condition> conditions = new ArrayList<>();

        for (var entry : requestsWithId.entrySet()) {
            conditions.add(
                    DSL.and(BANNERS_ADDITIONS.BID.eq(entry.getKey().getBid()),
                            BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(disclaimer))
            );
        }

        configuration.dsl()
                .deleteFrom(BANNERS_ADDITIONS)
                .where(DSL.or(conditions))
                .execute();
    }

    public void linkToBanners(Configuration configuration, Map<DisclaimerUpdateRequest, Long> requestsWithId) {
        var step1 = configuration.dsl().insertInto(BANNERS_ADDITIONS)
                .columns(BANNERS_ADDITIONS.BID, BANNERS_ADDITIONS.ADDITIONS_ITEM_ID, BANNERS_ADDITIONS.ADDITIONS_TYPE,
                        BANNERS_ADDITIONS.SEQUENCE_NUM);

        for (var req : requestsWithId.entrySet()) {
            DisclaimerUpdateRequest disclaimerUpdateRequest = req.getKey();
            long itemId = req.getValue();
            step1 = step1.values(disclaimerUpdateRequest.getBid(), itemId, disclaimer, 1L);
        }

        step1.execute();
    }

    public Map<String, Long> getExistedRecords(Configuration conf,
                                               Collection<DisclaimerUpdateRequest> requests) {

        List<Condition> conditions = new ArrayList<>();

        for (var req : requests) {
            conditions.add(
                    DSL.and(ADDITIONS_ITEM_DISCLAIMERS.CLIENT_ID.eq(req.getClientId()),
                            ADDITIONS_ITEM_DISCLAIMERS.HASH.eq(ULong.valueOf(req.getHash())))
            );
        }

        var existedKeys = conf.dsl()
                .selectDistinct(DSL.concat(ADDITIONS_ITEM_DISCLAIMERS.CLIENT_ID, val("/"),
                        ADDITIONS_ITEM_DISCLAIMERS.HASH).as("uniq_key"),
                        ADDITIONS_ITEM_DISCLAIMERS.ADDITIONS_ITEM_ID
                )
                .from(ADDITIONS_ITEM_DISCLAIMERS)
                .where(DSL.or(conditions)).fetch();

        return existedKeys.stream().collect(Collectors.toMap(Record2::value1, Record2::value2, (a, b) -> a));
    }

    public Map<DisclaimerUpdateRequest, Long> insertNewDisclaimers(Configuration conf,
                                                                   Collection<DisclaimerUpdateRequest> requests) {

        if (requests.isEmpty()) {
            return Map.of();
        }

        List<Long> ids = shardHelper.generateAdditionItemIds(requests.size());

        var step1 = conf.dsl().insertInto(ADDITIONS_ITEM_DISCLAIMERS)
                .columns(ADDITIONS_ITEM_DISCLAIMERS.ADDITIONS_ITEM_ID,
                        ADDITIONS_ITEM_DISCLAIMERS.HASH,
                        ADDITIONS_ITEM_DISCLAIMERS.CLIENT_ID, ADDITIONS_ITEM_DISCLAIMERS.DISCLAIMER_TEXT);

        Map<DisclaimerUpdateRequest, Long> result = new HashMap<>();
        int idPos = 0;

        for (var req : requests) {
            result.put(req, ids.get(idPos));

            step1 = step1.values(ids.get(idPos++), ULong.valueOf(req.getHash()), req.getClientId(),
                    req.getDisclaimerText());
        }

        step1.execute();

        return result;
    }

}
