package ru.yandex.direct.core.entity.moderation.service.receiving.dyn_disclaimers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jooq.Configuration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import static java.util.stream.Collectors.toMap;

@Service
public class DisclaimersUpdatingService {
    private final static Long EMPTY_DISCLAIMER_ID = Long.MIN_VALUE;
    private final DisclaimersRepository disclaimersRepository;

    @Autowired
    public DisclaimersUpdatingService(DisclaimersRepository disclaimersRepository) {
        this.disclaimersRepository = disclaimersRepository;
    }

    public void update(Configuration configuration, List<DisclaimerUpdateRequest> requests) {
        Map<DisclaimerUpdateRequest, Long> requestsWithId = saveRequests(configuration, requests);
        linkToBanners(configuration, requestsWithId);
    }

    private void linkToBanners(Configuration configuration, Map<DisclaimerUpdateRequest, Long> requestsWithId) {
        if (requestsWithId.isEmpty()) {
            return;
        }

        disclaimersRepository.removeBannerLinks(configuration, requestsWithId);

        Map<DisclaimerUpdateRequest, Long> requestsWithNonEmptyDisclaimers =
                requestsWithId.entrySet()
                        .stream()
                        .filter(e -> !e.getValue().equals(EMPTY_DISCLAIMER_ID))
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

        disclaimersRepository.linkToBanners(configuration, requestsWithNonEmptyDisclaimers);
    }

    private List<DisclaimerUpdateRequest> separateRequestsWithEmptyDisclaimers(List<DisclaimerUpdateRequest> requests,
                                                                               Map<DisclaimerUpdateRequest, Long> requestLongMap) {

        List<DisclaimerUpdateRequest> nonEmptyRequests = new ArrayList<>();

        for (var req : requests) {
            if (req.isDisclaimerEmpty()) {
                requestLongMap.put(req, EMPTY_DISCLAIMER_ID);
            } else {
                nonEmptyRequests.add(req);
            }
        }

        return nonEmptyRequests;
    }

    private List<DisclaimerUpdateRequest> separateExistedDisclaimers(Configuration configuration,
                                                                     List<DisclaimerUpdateRequest> requests,
                                                                     Map<DisclaimerUpdateRequest, Long> requestLongMap) {
        List<DisclaimerUpdateRequest> newRequests = new ArrayList<>();
        Map<String, Long> existedRecords = disclaimersRepository.getExistedRecords(configuration, requests);

        for (var req : requests) {
            Long id = existedRecords.get(req.uniqKey());
            if (id != null) {
                requestLongMap.put(req, id);
            } else {
                newRequests.add(req);
            }
        }

        return newRequests;
    }

    private Map<DisclaimerUpdateRequest, Long> saveRequests(Configuration configuration,
                                                            List<DisclaimerUpdateRequest> requests) {

        Map<DisclaimerUpdateRequest, Long> requestLongMap = new HashMap<>();

        List<DisclaimerUpdateRequest> nonEmptyRequests = separateRequestsWithEmptyDisclaimers(requests, requestLongMap);

        List<DisclaimerUpdateRequest> newRequests = separateExistedDisclaimers(configuration, nonEmptyRequests,
                requestLongMap);

        Map<DisclaimerUpdateRequest, Long> insertResult = disclaimersRepository.insertNewDisclaimers(configuration,
                newRequests);

        return Stream.of(requestLongMap, insertResult)
                .map(Map::entrySet)
                .flatMap(Set::stream)
                .collect(toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

}
