package ru.yandex.direct.core.entity.moderation.service.receiving.operations.banners;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import org.jooq.Configuration;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.moderation.model.ModerationDecision;
import ru.yandex.direct.core.entity.moderation.model.callout.CalloutModerationResponse;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderation.service.receiving.operations.ModerationResponseProcessingOp;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDITIONS_ITEM_CALLOUTS;
import static ru.yandex.direct.dbschema.ppc.enums.AdditionsItemCalloutsLastmoderationstatus.Accepted;
import static ru.yandex.direct.dbschema.ppc.enums.AdditionsItemCalloutsLastmoderationstatus.Rejected;
import static ru.yandex.direct.dbschema.ppc.enums.AdditionsItemCalloutsStatusmoderate.No;
import static ru.yandex.direct.dbschema.ppc.enums.AdditionsItemCalloutsStatusmoderate.Yes;

@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class SetCalloutStatusModerateOp implements ModerationResponseProcessingOp<CalloutModerationResponse> {

    private final Map<Long, ModerationDecision> calloutIdToModerationDecisions = new HashMap<>();

    @Override
    public void consume(BulkUpdateHolder bulkUpdateHolder, CalloutModerationResponse response) {
        calloutIdToModerationDecisions.put(response.getMeta().getCalloutId(), response.getResult().getVerdict());
    }

    @Override
    public void flush(Configuration configuration, BulkUpdateHolder bulkUpdateHolder) {
        Map<ModerationDecision, List<Long>> decisionToIdMap =
                EntryStream.of(calloutIdToModerationDecisions)
                        .invert()
                        .grouping();

        acceptCallouts(configuration, decisionToIdMap.get(ModerationDecision.Yes));
        declineCallouts(configuration, decisionToIdMap.get(ModerationDecision.No));
    }

    private void acceptCallouts(Configuration configuration, Collection<Long> acceptedCalloutIds) {
        if (isEmpty(acceptedCalloutIds)) {
            return;
        }
        configuration.dsl()
                .update(ADDITIONS_ITEM_CALLOUTS)
                .set(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE, Yes)
                .set(ADDITIONS_ITEM_CALLOUTS.LAST_MODERATION_STATUS, Accepted)
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(acceptedCalloutIds))
                .execute();
    }

    private void declineCallouts(Configuration configuration, Collection<Long> declinedCalloutIds) {
        if (isEmpty(declinedCalloutIds)) {
            return;
        }
        configuration.dsl()
                .update(ADDITIONS_ITEM_CALLOUTS)
                .set(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE, No)
                .set(ADDITIONS_ITEM_CALLOUTS.LAST_MODERATION_STATUS, Rejected)
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(declinedCalloutIds))
                .execute();
    }
}
