package ru.yandex.direct.core.entity.moderation.service.receiving.operations.banners;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationResponse;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.regions.GeoTree;

import static java.util.function.Predicate.not;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultPriority;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Останавливает прайсовый баннер если он принадлежит дефолтной группе и его минус-регионы пересекаются с гео кампании.
 * Это нужно, чтобы гарантировать полноту прайсовой кампании.
 */
@ParametersAreNonnullByDefault
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class StopCpmPriceBannerIfMinusRegionsOverlapOp extends BulkBannersOp {

    private final BannerTypedRepository bannerTypedRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final PricePackageService pricePackageService;

    private final List<BannerModerationResponse> responses = new ArrayList<>();

    @Autowired
    public StopCpmPriceBannerIfMinusRegionsOverlapOp(
            BannerTypedRepository bannerTypedRepository,
            CampaignRepository campaignRepository,
            CampaignTypedRepository campaignTypedRepository,
            AdGroupRepository adGroupRepository,
            PricePackageService pricePackageService) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.pricePackageService = pricePackageService;
    }

    @Override
    public void consume(BulkUpdateHolder bulkUpdateHolder, BannerModerationResponse response) {
        // для ответов с пустыми минус-регионами эта операция ничего делать не должна
        // экономим походы в базу и выкидываем такие ответы
        if (!isEmpty(response.getResult().getMinusRegions())) {
            responses.add(response);
        }
    }

    @Override
    public void flush(Configuration configuration, BulkUpdateHolder bulkUpdateHolder) {
        if (isEmpty(responses)) {
            return;
        }

        DSLContext ctx = DSL.using(configuration);

        Set<Long> bannerIds = responses.stream()
                .map(response -> response.getMeta().getBannerId())
                .collect(Collectors.toSet());
        // не используем adGroupId и campaignId из meta, т.к. нет уверенности, что модерация их всегда присылает
        var banners = bannerTypedRepository.getSafely(ctx, bannerIds, BannerWithSystemFields.class);
        Map<Long, BannerWithSystemFields> bannersMap = listToMap(banners, BannerWithSystemFields::getId);
        // На всякий случай защитимся от NPE - баннеры, которых нет в базе обрабатывать не нужно.
        // (вдруг баннер удалили, а потом долетел ответ модерации?)
        removeNonexistentBanners(bannerIds, bannersMap);

        Set<Long> campaignIds = listToSet(banners, BannerWithSystemFields::getCampaignId);
        Map<Long, CampaignType> campaignTypeMap = campaignRepository.getCampaignsTypeMap(ctx, campaignIds);
        removeNotCpmPriceCampaigns(bannerIds, bannersMap, campaignTypeMap);

        Set<Long> cpmPriceAdGroupIds = listToSet(banners, BannerWithSystemFields::getAdGroupId);
        Map<Long, Long> cpmPriceAdGroupPriorities = adGroupRepository.getAdGroupsPriority(ctx, cpmPriceAdGroupIds);
        removeNotDefaultAdgroups(bannerIds, bannersMap, cpmPriceAdGroupPriorities);

        Set<Long> reachableCampaignIds = bannerIds.stream()
                .map(bannersMap::get)
                .map(BannerWithSystemFields::getCampaignId)
                .collect(Collectors.toSet());
        removeUnreachableCampaigns(campaignTypeMap, reachableCampaignIds);

        Map<Long, CpmPriceCampaign> cpmPriceCampaigns = campaignTypedRepository.getTypedCampaignsMap(
                ctx, campaignTypeMap, CampaignType.CPM_PRICE);
        List<Long> bannersToReset = responses.stream()
                .filter(response -> bannerIds.contains(response.getMeta().getBannerId()))
                .filter(response -> bannerMinusRegionsOverlapWithCampaign(response,
                        cpmPriceCampaigns.get(bannersMap.get(response.getMeta().getBannerId()).getCampaignId())))
                .map(response -> response.getMeta().getBannerId())
                .collect(Collectors.toList());

        for (var bid : bannersToReset) {
            bulkUpdateHolder.get(BANNERS.BID)
                    .forId(bid)
                    .set(BANNERS.STATUS_SHOW, BannersStatusshow.No);
        }

        responses.clear();
    }

    private static void removeNonexistentBanners(Set<Long> bannerIds, Map<Long, BannerWithSystemFields> bannersMap) {
        bannerIds.removeIf(not(bannersMap::containsKey));
    }

    private static void removeNotCpmPriceCampaigns(Collection<Long> bannerIds,
                                                   Map<Long, BannerWithSystemFields> bannersMap,
                                                   Map<Long, CampaignType> campaignTypeMap) {
        bannerIds.removeIf(bannerId -> {
            var campaignId = bannersMap.get(bannerId).getCampaignId();
            return !CampaignType.CPM_PRICE.equals(campaignTypeMap.get(campaignId));
        });
    }

    private static void removeNotDefaultAdgroups(Collection<Long> bannerIds,
                                                 Map<Long, BannerWithSystemFields> bannersMap,
                                                 Map<Long, Long> cpmPriceAdGroupPriorities) {
        bannerIds.removeIf(bannerId -> {
            var adGroupId = bannersMap.get(bannerId).getAdGroupId();
            return !isDefaultPriority(cpmPriceAdGroupPriorities.get(adGroupId));
        });
    }

    private static void removeUnreachableCampaigns(Map<Long, CampaignType> campaignTypeMap,
                                                   Set<Long> reachableCampaignIds) {
        campaignTypeMap.entrySet().removeIf(e -> !reachableCampaignIds.contains(e.getKey()));
    }

    private boolean bannerMinusRegionsOverlapWithCampaign(BannerModerationResponse response,
                                                          CpmPriceCampaign cpmPriceCampaign) {
        List<Long> minusRegions = response.getResult().getMinusRegions();
        if (isEmpty(minusRegions)) {
            // geoTree.isAnyRegionOrSubRegionIncludedIn возвращает true на пустой minusRegionsSet
            // нам этом не подходит, поэтому делаем здесь return false
            return false;
        }
        Set<Long> campaignRegionsSet = new HashSet<>(cpmPriceCampaign.getFlightTargetingsSnapshot().getGeoExpanded());
        Set<Long> minusRegionsSet = new HashSet<>(minusRegions);
        return getGeoTree().isAnyRegionOrSubRegionIncludedIn(minusRegionsSet, campaignRegionsSet);
    }

    private GeoTree getGeoTree() {
        return pricePackageService.getGeoTree();
    }
}
