package ru.yandex.direct.core.entity.moderation.service.receiving.operations.banners;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationResponse;
import ru.yandex.direct.core.entity.moderation.model.BaseModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.Verdict;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderation.service.receiving.operations.common.InsertOrUpdateModerationReasonsOp;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReason;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonDetailed;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonStatusModerate;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonStatusPostModerate;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonStatusSending;
import ru.yandex.direct.core.entity.moderationreason.repository.ModerationReasonRepository;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbutil.model.ClientId;

import static java.util.function.Predicate.not;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.feature.FeatureName.DECLINE_UNFAMILY_MCBANNER_MODERATION;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Останавливает баннер на поиске, если среди флагов модерации есть флаг unfamily
 */
@ParametersAreNonnullByDefault
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class StopUnfamilyMCBannerOp extends BulkBannersOp {

    private final BannerTypedRepository bannerTypedRepository;
    private final CampaignRepository campaignRepository;
    private final ModerationReasonRepository moderationReasonRepository;
    private final List<BannerModerationResponse> responses = new ArrayList<>();
    private final FeatureService featureService;

    @Autowired
    public StopUnfamilyMCBannerOp(
            BannerTypedRepository bannerTypedRepository,
            CampaignRepository campaignRepository,
            ModerationReasonRepository moderationReasonRepository,
            FeatureService featureService) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.campaignRepository = campaignRepository;
        this.moderationReasonRepository = moderationReasonRepository;
        this.featureService = featureService;
    }

    @Override
    public void consume(BulkUpdateHolder bulkUpdateHolder, BannerModerationResponse response) {
        Boolean declineUnfamilyBannerFeature = Optional.ofNullable(response.getMeta())
                .map(BaseModerationMeta::getClientId)
                .map(ClientId::fromLong)
                .map(clientId -> featureService.isEnabledForClientId(clientId, DECLINE_UNFAMILY_MCBANNER_MODERATION))
                .orElse(false);
        if (declineUnfamilyBannerFeature && Optional.ofNullable(response.getResult())
                .map(Verdict::getFlags)
                .map(flags -> flags.containsKey(BannerFlags.UNFAMILY.getKey()))
                .orElse(false)) {
            responses.add(response);
        }
    }

    @Override
    public void flush(Configuration configuration, BulkUpdateHolder bulkUpdateHolder) {
        if (isEmpty(responses)) {
            return;
        }
        Set<Long> bannerIds = responses.stream()
                .map(response -> response.getMeta().getBannerId())
                .collect(Collectors.toSet());
        DSLContext ctx = DSL.using(configuration);
        var banners = bannerTypedRepository.getSafely(ctx, bannerIds, BannerWithSystemFields.class);
        Map<Long, BannerWithSystemFields> bannersMap = listToMap(banners, BannerWithSystemFields::getId);
        // На всякий случай защитимся от NPE - баннеры, которых нет в базе обрабатывать не нужно.
        // (вдруг баннер удалили, а потом долетел ответ модерации?)
        removeNonexistentBanners(bannerIds, bannersMap);

        Set<Long> campaignIds = listToSet(banners, BannerWithSystemFields::getCampaignId);

        Map<Long, CampaignType> campaignTypeMap = EntryStream.of(campaignRepository.getCampaignsTypeMap(ctx, campaignIds))
                .filterValues(ct -> ct.equals(CampaignType.MCBANNER))
                .toMap();
        removeBannersWithDifferentCampaign(bannerIds, bannersMap, campaignTypeMap);

        Map<Long, BannerModerationResponse> bidBannerModerationMap = listToMap(responses,
                resp -> resp.getMeta().getBannerId());

        List<ModerationReason> modReasons = new ArrayList<>();
        for (var bid : bannerIds) {
            bulkUpdateHolder.get(BANNERS.BID)
                    .forId(bid)
                    .set(BANNERS.STATUS_MODERATE, BannersStatusmoderate.No)
                    .set(BANNERS.STATUS_POST_MODERATE, BannersStatuspostmoderate.Rejected);
            var response =  bidBannerModerationMap.get(bid);
            List<ModerationReasonDetailed> reasons = nvl(response.getResult().getReasonsWithDetails(), new ArrayList<>());
            reasons.add(new ModerationReasonDetailed().withId(1256L));
            InsertOrUpdateModerationReasonsOp.ReasonsInfo reasonsInfo = new InsertOrUpdateModerationReasonsOp.ReasonsInfo(
                    response.getMeta().getBannerId(),
                    response.getMeta().getCampaignId(),
                    response.getMeta().getClientId(),
                    reasons
            );
            modReasons.add(createModerationReason(reasonsInfo));
        }

        moderationReasonRepository.insertOrUpdateModerationReasons(configuration, modReasons);
        responses.clear();
    }

    private void removeNonexistentBanners(Set<Long> bannerIds, Map<Long, BannerWithSystemFields> bannersMap) {
        bannerIds.removeIf(not(bannersMap::containsKey));
    }

    private void removeBannersWithDifferentCampaign(Collection<Long> bannerIds,
                                                    Map<Long, BannerWithSystemFields> bannersMap,
                                                    Map<Long, CampaignType> campaignTypeMap) {
        bannerIds.removeIf(bannerId -> {
            var campaignId = bannersMap.get(bannerId).getCampaignId();
            return !campaignTypeMap.containsKey(campaignId);
        });
    }

    private ModerationReason createModerationReason(InsertOrUpdateModerationReasonsOp.ReasonsInfo reasonsInfo) {
        return new ModerationReason()
                .withObjectId(reasonsInfo.getObjectId())
                .withObjectType(ModerationReasonObjectType.BANNER)
                .withClientId(reasonsInfo.getClientId())
                .withCampaignId(reasonsInfo.getCampaignId())
                .withStatusSending(ModerationReasonStatusSending.NO)
                .withStatusModerate(ModerationReasonStatusModerate.NO)
                .withStatusPostModerate(ModerationReasonStatusPostModerate.NO)
                .withCreatedAt(LocalDateTime.now())
                .withReasons(reasonsInfo.getReasons());
    }

}
