package ru.yandex.direct.core.entity.moderation.service.receiving.operations.banners;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.apache.commons.collections4.MapUtils;
import org.jooq.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.FlagProperty;
import ru.yandex.direct.core.entity.banner.type.flags.BannerFlagsRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationResponse;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderation.service.receiving.MinusRegionsToBannerFlagsService;
import ru.yandex.direct.dbschema.ppc.tables.Banners;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;

import static ru.yandex.direct.core.entity.banner.model.BannerFlags.MINUS_REGION_KZ;
import static ru.yandex.direct.core.entity.banner.model.BannerFlags.MINUS_REGION_RB;
import static ru.yandex.direct.core.entity.banner.model.BannerFlags.MINUS_REGION_RU;
import static ru.yandex.direct.core.entity.banner.model.BannerFlags.MINUS_REGION_TR;
import static ru.yandex.direct.core.entity.banner.model.BannerFlags.MINUS_REGION_UA;
import static ru.yandex.direct.core.entity.banner.model.BannerFlags.MINUS_REGION_UZ;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class UpdateBannerFlagsOp extends BulkBannersOp {
    private final Map<Long, BannerFlags> bidToFlags = new HashMap<>();
    private final BannerFlagsRepository bannerFlagsRepository;
    private static final List<FlagProperty> flagsWithValues =
            List.of(BannerFlags.AGE, BannerFlags.BABY_FOOD, BannerFlags.YA_PAGES);
    private static final String DEFAULT_VALUE = "default";
    private final FeatureService featureService;
    private final MinusRegionsToBannerFlagsService minusRegionsToBannerFlagsService;
    private Map<Long, Long> bid2ClientId = new HashMap<>();

    private static final Logger logger = LoggerFactory.getLogger(UpdateBannerFlagsOp.class);

    public static final Map<Long, FlagProperty<Boolean>> MINUS_REGION_FLAGS = Map.of(
            225L, MINUS_REGION_RU,
            977L, MINUS_REGION_RU,
            159L, MINUS_REGION_KZ,
            187L, MINUS_REGION_UA,
            149L, MINUS_REGION_RB,
            983L, MINUS_REGION_TR,
            171L, MINUS_REGION_UZ
    );

    @Autowired
    public UpdateBannerFlagsOp(BannerFlagsRepository bannerFlagsRepository,
                               FeatureService featureService,
                               MinusRegionsToBannerFlagsService minusRegionsToBannerFlagsService) {
        this.bannerFlagsRepository = bannerFlagsRepository;
        this.featureService = featureService;
        this.minusRegionsToBannerFlagsService = minusRegionsToBannerFlagsService;
    }

    @Override
    public void consume(BulkUpdateHolder bulkUpdateHolder, BannerModerationResponse object) {
        bid2ClientId.put(object.getMeta().getBannerId(), object.getMeta().getClientId());
        bidToFlags.put(object.getMeta().getBannerId(), getFlagsFromResponse(object));
    }

    public BannerFlags getFlagsFromResponse(BannerModerationResponse object) {
        Map<String, String> flags = new HashMap<>();
        if (object.getResult().getFlags() != null) {
            flags.putAll(object.getResult().getFlags());
        }

        if (minusRegionsToBannerFlagsService.isMinusRegionsToBannerFlagsSaveEnabled(object.getMeta().getCampaignId())) {
            flags.putAll(convertMinusRegionsToFlags(object.getResult().getMinusRegions()));
        }

        BannerFlags bannerFlags = new BannerFlags();

        if (!MapUtils.isEmpty(flags)) {
            bannerFlags.withFlagsFromModeration(flags);
        }
        return bannerFlags;
    }

    private Map<String, String> convertMinusRegionsToFlags(@Nullable List<Long> minusRegions) {
        if (minusRegions == null) {
            return Collections.emptyMap();
        }

        Map<String, String> flags = new HashMap<>();
        for (Long minusRegion : minusRegions) {
            if (!MINUS_REGION_FLAGS.containsKey(minusRegion)) {
                logger.warn("Unknown minus region value = {}", minusRegion);
            } else {
                flags.put(MINUS_REGION_FLAGS.get(minusRegion).getKey(), null);
            }
        }

        return flags;
    }

    Set<Long> getBidsWithDefaultValues() {
        Set<Long> bidsWithDefaultValues = new HashSet<>();

        for (var entry : bidToFlags.entrySet()) {
            for (var flag : flagsWithValues) {
                if (DEFAULT_VALUE.equals(entry.getValue().getFlags().get(flag.getKey()))) {
                    bidsWithDefaultValues.add(entry.getKey());
                }
            }
        }

        return bidsWithDefaultValues;
    }

    Map<Long, BannerFlags> loadFlagsWithValuesFromDb(Configuration configuration) {
        Set<Long> bidsWithDefaultValues = getBidsWithDefaultValues();

        return bannerFlagsRepository.getBannersWithFlags(configuration, bidsWithDefaultValues, flagsWithValues);
    }

    void replaceDefaultValuesByRealValuesFromDb(BannerFlags dbFlags, Map<String, String> incomingFlags) {
        for (FlagProperty flag : flagsWithValues) {

            if (!incomingFlags.containsKey(flag.getKey())) {
                continue;
            }

            String value;
            String dbValue = dbFlags != null ? dbFlags.getFlags().get(flag.getKey()) : null;
            String incomingValue = incomingFlags.get(flag.getKey());

            if (dbValue != null && DEFAULT_VALUE.equals(incomingValue)) {
                value = dbValue;
            } else if (dbValue != null && incomingValue != null) {
                value = incomingValue;
            } else if (dbValue == null && DEFAULT_VALUE.equals(incomingValue)) {
                value = flag.getDefaultValue();
            } else if (dbValue == null && incomingValue != null) {
                value = incomingValue;
            } else {
                value = dbValue;
            }

            incomingFlags.put(flag.getKey(), value);
        }
    }

    @Override
    public void flush(Configuration configuration, BulkUpdateHolder bulkUpdateHolder) {
        super.flush(configuration, bulkUpdateHolder);

        Map<Long, BannerFlags> flagsFromDb = loadFlagsWithValuesFromDb(configuration);

        var clientIds = bid2ClientId.values().stream().distinct().map(ClientId::fromLong).collect(Collectors.toSet());

        var clientId2feature = featureService.isEnabledForClientIdsOnlyFromDb(clientIds,
                FeatureName.SOCIAL_ADVERTISING.getName());

        for (var bid : bidToFlags.keySet()) {
            replaceDefaultValuesByRealValuesFromDb(nvl(flagsFromDb.get(bid), new BannerFlags()),
                    bidToFlags.get(bid).getFlags());

            if (clientId2feature.get(ClientId.fromLong(bid2ClientId.get(bid)))) {
                bidToFlags.get(bid).with(BannerFlags.SOCIAL_ADVERTISING, true);
            }

            bulkUpdateHolder
                    .get(BANNERS.BID)
                    .forId(bid)
                    .set(Banners.BANNERS.FLAGS, BannerFlags.toSource(bidToFlags.get(bid)));
        }
    }
}
